package com.phonepe.sdk.javasdk.transaction.init;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.phonepe.sdk.javasdk.config.URLCreator;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.enums.ConfigType;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.checksum.ChecksumGenerator;
import com.phonepe.sdk.javasdk.transaction.client.TransactionCommand;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.models.InitParams;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreator;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import lombok.extern.slf4j.Slf4j;
import okhttp3.HttpUrl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Template Method class for initiating the transaction for any given type of flow.
 */
@Slf4j
public abstract class TransactionInitiator {

    protected TransactionCommand transactionCommand;
    protected InitConfig initConfig;
    protected MerchantConfig merchantConfig;
    protected ChecksumGenerator checksumGenerator;
    protected PhonePeHttpRequestCreator<InitRequest> phonePeHttpRequestCreator;
    protected URLCreator urlCreator;
    protected HttpUrl httpUrl;

    public TransactionInitiator(final TransactionCommand transactionCommand,
                                final InitConfig initConfig,
                                final MerchantConfig merchantConfig,
                                final ChecksumGenerator checksumGenerator,
                                final URLCreator urlCreator,
                                final PhonePeHttpRequestCreator<InitRequest> phonePeHttpRequestCreator) {
        this.transactionCommand = transactionCommand;
        this.initConfig = initConfig;
        this.merchantConfig = merchantConfig;
        this.checksumGenerator = checksumGenerator;
        this.phonePeHttpRequestCreator = phonePeHttpRequestCreator;
        this.urlCreator = urlCreator;
        this.httpUrl = urlCreator.getHttpUrl(ConfigType.INIT, initConfig.getOverrideHost(), initConfig.getApiVersion().getValue());
    }

    public InitResponse initiateTransaction(final InitParams initParams,
                                            final int keyIndex) throws PhonePeClientException {

        try {
            final PhonePeHttpRequest<?> initRequest = buildInitRequest(this.merchantConfig.getMerchantId(),
                                                                       initParams);
            final String apiKey = KeyUtils.getAPIKeyFromIndex(this.merchantConfig.getApiKeys(),
                                                              keyIndex);
            final List<String> params = getParamsList(initParams.getTransactionId(),
                                                      initParams.getAmount());
            final String checksumHeaderValue = this.checksumGenerator.getChecksumValue(initRequest,
                                                                                       this.httpUrl.encodedPath(),
                                                                                       params,
                                                                                       apiKey,
                                                                                       keyIndex);
            return initiateTransaction(initRequest,
                                       initParams,
                                       checksumHeaderValue);
        } catch (PhonePeClientException e) {
            log.error("Error while making call to PhonePe init transaction API");
            throw e;
        } catch (Exception e) {
            log.error("Error in calling PhonePe init transaction API",
                      e);
            Map<String, Object> objectMap = new HashMap<>();
            objectMap.put("MESSAGE",
                          e.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,
                                             "Error executing initiate transaction" + ": " + e.getMessage(),
                                             objectMap,
                                             e);
        }
    }

    private List<String> getParamsList(final String transactionId,
                                       final Long amount) {
        List<String> paramsList = new ArrayList<>();
        paramsList.add(this.merchantConfig.getMerchantId());
        paramsList.add(transactionId);
        paramsList.add(Long.toString(amount));
        return paramsList;
    }

    private InitResponse initiateTransaction(final PhonePeHttpRequest<?> phonePeHttpRequest,
                                             final InitParams initParams,
                                             final String checksumHeader) throws PhonePeClientException {

        try {
            final List<HttpHeaderPair> httpHeaders = HttpUtils.getHttpHeaders(checksumHeader,
                                                                              this.merchantConfig.getProviderId(),
                                                                              initParams.getRedirectURL(),
                                                                              initParams.getRedirectMode(),
                                                                              initParams.getCallbackURL(),
                                                                              initParams.getCallbackMode());
            return this.transactionCommand.initTransaction(phonePeHttpRequest,
                                                           httpHeaders,
                                                           this.httpUrl);
        } catch (PhonePeClientException e) {
            log.error("Error while making call to PhonePe init transaction API");
            throw e;
        } catch (Exception e) {
            log.error("Error in calling PhonePe init transaction API",
                      e);
            Map<String, Object> objectMap = new HashMap<>();
            objectMap.put("MESSAGE",
                          e.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,
                                             "Error executing initiate transaction" + ": " + e.getMessage(),
                                             objectMap,
                                             e);
        }
    }

    protected abstract PhonePeHttpRequest<?> buildInitRequest(final String merchantId,
                                                              final InitParams initParams) throws JsonProcessingException;


}
