package com.phonepe.sdk.javasdk;

import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.ConfigurationFactory;
import com.phonepe.sdk.javasdk.config.DefaultEndpointCreator;
import com.phonepe.sdk.javasdk.config.DefaultUrlCreator;
import com.phonepe.sdk.javasdk.config.URLCreator;
import com.phonepe.sdk.javasdk.config.models.DefaultConfig;
import com.phonepe.sdk.javasdk.config.models.Endpoint;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.PhonePeConfig;
import com.phonepe.sdk.javasdk.config.models.SDKConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.exception.PhonePeConfigurationException;
import com.phonepe.sdk.javasdk.http.PhonePeHttpClientFactory;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler;
import com.phonepe.sdk.javasdk.transaction.callback.StatusResponseHandler;
import com.phonepe.sdk.javasdk.transaction.checksum.ChecksumGenerator;
import com.phonepe.sdk.javasdk.transaction.checksum.ChecksumGeneratorFactory;
import com.phonepe.sdk.javasdk.transaction.client.TransactionCommand;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiatorFactory;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.models.InitParams;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreator;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreatorFactory;
import com.phonepe.sdk.javasdk.transaction.status.PaymentStateCreator;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;

import java.io.IOException;


/**
 * Facade that wraps the {@link com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator} and
 * {@link com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler} and
 * {@link com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker} for the
 * user.
 */
@Slf4j
public class PhonePeClient {

    private TransactionInitiator transactionInitiator;
    private CallbackHandler callbackHandler;
    private TransactionStatusChecker statusChecker;
    private int defaultKeyIndex;

    @Builder
    public PhonePeClient(final SDKConfig sdkConfig) throws PhonePeConfigurationException,IOException {
        Preconditions.checkNotNull(sdkConfig);
        init(sdkConfig);
    }

    private void init(final SDKConfig config) throws PhonePeConfigurationException, IOException{
        init(config, null);
    }

    private void init(final SDKConfig sdkConfig, final OkHttpClient okHttpClient) throws PhonePeConfigurationException, IOException {
        final PhonePeConfig phonePeConfig = sdkConfig.getPhonePeConfig();
        final DefaultConfig defaultConfig = ConfigurationFactory.<DefaultConfig>builder()
                .clazz(DefaultConfig.class)
                .build()
                .buildConfig("java_sdk_default.yml");
        final HttpClientConfig httpClientConfig = phonePeConfig.getHttpClientConfig();
        final Endpoint endpoint = phonePeConfig.getEndpoint();
        final StatusConfig statusConfig = phonePeConfig.getStatusConfig();
        final MerchantConfig merchantConfig = sdkConfig.getMerchantConfig();
        final InitConfig initConfig = phonePeConfig.getInitConfig();
        final ChecksumGeneratorFactory checksumGeneratorFactory = new ChecksumGeneratorFactory();
        final DefaultEndpointCreator defaultEndpointCreator = DefaultEndpointCreator.builder()
                                                                                    .defaultConfig(defaultConfig)
                                                                                    .build();

        final DefaultUrlCreator defaultUrlCreator = DefaultUrlCreator.builder()
                                                                     .defaultConfig(defaultConfig)
                                                                     .build();
        final URLCreator urlCreator = URLCreator.builder()
                                                .defaultEndpointCreator(defaultEndpointCreator)
                                                .endpoint(endpoint)
                                                .defaultUrlCreator(defaultUrlCreator)
                                                .build();

        final PaymentStateCreator paymentStateCreator = new PaymentStateCreator();
        final OkHttpClient client = PhonePeHttpClientFactory.buildOkHttpClient(okHttpClient,
                                                                               httpClientConfig,
                                                                               HttpUtils.isHttpEndpointSecured(endpoint));
        final TransactionCommand transactionCommand = new TransactionCommand(client, defaultConfig);
        final ChecksumGenerator statusChecksumGenerator = statusConfig
                .getApiVersion()
                .accept(checksumGeneratorFactory);
        final ChecksumGenerator initChecksumGenerator = initConfig
                .getApiVersion()
                .accept(checksumGeneratorFactory);

        final PhonePeHttpRequestCreatorFactory<InitRequest> phonePeHttpRequestCreatorFactory =
                new PhonePeHttpRequestCreatorFactory<>();
        final PhonePeHttpRequestCreator<InitRequest> phonePeHttpRequestCreator = initConfig
                .getApiVersion()
                .accept(phonePeHttpRequestCreatorFactory);


        final TransactionInitiatorFactory transactionInitiatorFactory = TransactionInitiatorFactory
                .builder()
                .merchantConfig(merchantConfig)
                .initConfig(initConfig)
                .transactionCommand(transactionCommand)
                .checksumGenerator(initChecksumGenerator)
                .urlCreator(urlCreator)
                .phonePeHttpRequestCreator(phonePeHttpRequestCreator)
                .build();
        this.statusChecker = TransactionStatusChecker
                .builder()
                .merchantConfig(merchantConfig)
                .statusConfig(statusConfig)
                .transactionCommand(transactionCommand)
                .checksumGenerator(statusChecksumGenerator)
                .paymentStateCreator(paymentStateCreator)
                .urlCreator(urlCreator)
                .build();

        this.callbackHandler = CallbackHandler
                .builder()
                .merchantConfig(merchantConfig)
                .transactionStatusChecker(this.statusChecker)
                .build();

        this.transactionInitiator = initConfig
                .getInitType()
                .accept(transactionInitiatorFactory);
        this.defaultKeyIndex = KeyUtils.getDefaultKeyIndex(merchantConfig.getApiKeys());
    }

    public InitResponse initTransaction(final InitParams initParam) throws PhonePeClientException {
        return initTransaction(initParam,
                               defaultKeyIndex);
    }

    public InitResponse initTransaction(final InitParams initParams,
                                        final int keyIndex) throws PhonePeClientException {
        return this.transactionInitiator.initiateTransaction(initParams,
                                                             keyIndex);
    }

    public void handleCallback(final String responseReceived,
                               final String checksum,
                               final long transactionAmount,
                               final StatusResponseHandler customResponseHandler) throws PhonePeClientException {
         this.callbackHandler.handleCallback(responseReceived,
                                             checksum,
                                             transactionAmount,
                                             customResponseHandler);
    }

    public StatusResponse checkTransactionStatus(final String transactionId) throws PhonePeClientException {
        return checkTransactionStatus(transactionId,
                                      defaultKeyIndex);
    }

    public StatusResponse checkTransactionStatus(final String transactionId,
                                                 final int keyIndex) throws PhonePeClientException {
        return this.statusChecker.checkTransactionStatus(transactionId,
                                                         keyIndex);
    }

}