package com.phonepe.sdk.javasdk.config;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.TreeTraversingParser;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.SDKConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeConfigurationException;
import com.phonepe.sdk.javasdk.utils.MapperUtils;
import lombok.Builder;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;

public class ConfigurationFactory {

  private final ObjectMapper objectMapper;
  private final Validator validator;
  private final JsonFactory parserFactory;
  private static final String DEFAULT_SDK_CONFIG_FILE = "phonepe_java_sdk_properties.yml";

  @Builder
  public ConfigurationFactory() {
    this.parserFactory = new YAMLFactory();
    this.validator =  Validation.buildDefaultValidatorFactory()
                                .getValidator();
    this.objectMapper = MapperUtils.getMapper();
  }

  public SDKConfig buildConfig() throws PhonePeConfigurationException {
    return buildConfig(DEFAULT_SDK_CONFIG_FILE);
  }

  public SDKConfig buildConfig(final String fileName) throws PhonePeConfigurationException {
    Preconditions.checkNotNull(fileName);
    File file = null;
    try {
      file = new File(fileName);
      InputStream inputStream = getClass().getClassLoader().getResourceAsStream(fileName);
      final JsonNode node = objectMapper.readTree(createParser(inputStream));
      final String filename = file.toString();
      return build(node, filename);
    } catch (Exception exception) {
      throw new PhonePeConfigurationException(fileName, exception.getMessage());
    }
  }

  protected JsonParser createParser(InputStream inputStream) throws IOException {
    return parserFactory.createParser(inputStream);
  }

  private SDKConfig build(JsonNode node, String filename) throws IOException, PhonePeConfigurationException {
    final SDKConfig config = objectMapper.readValue(new TreeTraversingParser(node), SDKConfig.class);
    validate(filename, config);
    return config;
  }

  private void validate(String file, SDKConfig config) throws PhonePeConfigurationException {
    if (validator != null) {
      final Set<ConstraintViolation<SDKConfig>> violations = validator.validate(config);
      if (!violations.isEmpty()) {
        throw new PhonePeConfigurationException(file, violations);
      }
    }
  }
}