package com.phonepe.sdk.javasdk.transaction.status;

import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.Retryer;
import com.google.common.base.Predicate;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.checksum.payload.ChecksumGenerator;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.enums.PaymentState;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import com.phonepe.sdk.javasdk.utils.RetryUtils;
import lombok.Builder;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

@Slf4j
public class TransactionStatusChecker {

    private TransactionClient transactionClient;
    private StatusConfig statusConfig;
    private MerchantConfig merchantConfig;
    @Getter
    private ChecksumGenerator checksumGenerator;
    private String apiURLString;

    private static String PAYMENT_PENDING = "PAYMENT_PENDING";
    private static String INTERNAL_SERVER_ERROR = "INTERNAL_SERVER_ERROR";
    private static String PAYMENT_SUCCESS = "PAYMENT_SUCCESS";
    private static String PAYMENT_ERROR = "PAYMENT_ERROR";
    private static String PAYMENT_DECLINED = "PAYMENT_DECLINED";
    private static String PAYMENT_CANCELLED = "PAYMENT_CANCELLED";
    private static String TRANSACTION_NOT_FOUND = "TRANSACTION_NOT_FOUND";


    @Builder
    public TransactionStatusChecker(final TransactionClient transactionClient,
                                    final MerchantConfig merchantConfig,
                                    final StatusConfig statusConfig,
                                    final ChecksumGenerator checksumGenerator) {
        this.transactionClient = transactionClient;
        this.merchantConfig = merchantConfig;
        this.statusConfig = statusConfig;
        this.checksumGenerator = checksumGenerator;
        this.apiURLString = String.format("/%s/",this.statusConfig.getApiVersion().getValue())
                                  .concat("transaction/%s/%s/status");
    }

    public StatusResponse checkTransactionStatus(final String transactionId,
                                                 final int keyIndex) throws PhonePeClientException {
        try{
            final String apiURL = getAPIURL(transactionId);
            final String apiKey = KeyUtils.getAPIKeyFromIndex(this.merchantConfig.getApiKeys(),keyIndex);
            final List<String> params = getParamsList(transactionId);
            final String checksumHeaderValue = this.checksumGenerator.getChecksumValue(null, apiURL, params, apiKey, keyIndex);
            List<HttpHeaderPair> httpHeaders = HttpUtils.getHttpHeaders(checksumHeaderValue);
            Callable<PhonePeHttpResponse<TransactionStatusResponse>> transactionStatusCallable = getCallable(apiURL,
                                                                                                             httpHeaders);
            Predicate<PhonePeHttpResponse<TransactionStatusResponse>> phonePeHttpResponsePredicate = getRetryPredicate();
            Retryer<PhonePeHttpResponse<TransactionStatusResponse>> retryer = RetryUtils.getRetryer(statusConfig.getRetryConfig(),
                                                                                                    phonePeHttpResponsePredicate);
            PhonePeHttpResponse<TransactionStatusResponse> phonePeHttpResponse = retryer.call(transactionStatusCallable);
            return buildStatusResponse(phonePeHttpResponse);
        } catch (RetryException ex){
            PhonePeHttpResponse<TransactionStatusResponse> phonePeHttpResponse = (PhonePeHttpResponse<TransactionStatusResponse>) ex
                    .getLastFailedAttempt()
                    .getResult();
            return buildStatusResponse(phonePeHttpResponse);
        }
        catch (Exception ex){
            log.error("Exception occurred while performing transaction status check");
            Map<String, Object> objectMap = new HashMap<String, Object>();
            objectMap.put("MESSAGE", ex.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.EXECUTION_ERROR,"Error executing initiate transaction"
                                                                                              + ": " + ex.getMessage(), objectMap, ex);
        }
    }

    private List<String> getParamsList(String transactionId){
        List<String> paramsList =  new ArrayList<String>();
        paramsList.add(this.merchantConfig.getMerchantId());
        paramsList.add(transactionId);
        return paramsList;
    }

    private String getAPIURL(final String transactionId){
        return String.format(this.apiURLString,
                             this.merchantConfig.getMerchantId(),
                             transactionId);
    }


    private Callable<PhonePeHttpResponse<TransactionStatusResponse>> getCallable(final String apiURL,
                                                                                 final List<HttpHeaderPair> headerPairs){
        return  new Callable<PhonePeHttpResponse<TransactionStatusResponse>>() {
            @Override
            public PhonePeHttpResponse<TransactionStatusResponse> call() throws Exception {
                return transactionClient.getTransactionStatus(headerPairs, apiURL);
            }
        };
    }

    private Predicate<PhonePeHttpResponse<TransactionStatusResponse>> getRetryPredicate(){
        return new Predicate<PhonePeHttpResponse<TransactionStatusResponse>>(){
            @Override
            public boolean apply(PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse) {
                return transactionStatusResponse.getCode().equalsIgnoreCase(PAYMENT_PENDING) ||
                       transactionStatusResponse.getCode().equalsIgnoreCase(INTERNAL_SERVER_ERROR);
            }
        };
    }

    private StatusResponse buildStatusResponse(final PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse){
        return StatusResponse.builder()
                             .transactionId(transactionStatusResponse.getData().getTransactionId())
                             .paymentState(getPaymentState(transactionStatusResponse.getCode()))
                             .providerReferenceId(transactionStatusResponse.getData().getProviderReferenceId())
                             .amount(transactionStatusResponse.getData().getAmount())
                             .paidAmount(transactionStatusResponse.getData().getPaidAmount())
                             .paymentModes(ObjectUtils.isNotEmpty(transactionStatusResponse.getData().getPaymentModes())
                                           ?
                                           transactionStatusResponse.getData().getPaymentModes()
                                           :
                                           null)
                             .build();
    }

    private PaymentState getPaymentState(final String statusCode){
        if(PAYMENT_SUCCESS.equalsIgnoreCase(statusCode)){
            return PaymentState.PAYMENT_SUCCESS;
        }
        if(PAYMENT_ERROR.equalsIgnoreCase(statusCode) ||
           PAYMENT_DECLINED.equalsIgnoreCase(statusCode) ||
           PAYMENT_CANCELLED.equalsIgnoreCase(statusCode) ||
           TRANSACTION_NOT_FOUND.equalsIgnoreCase(statusCode)){
            return PaymentState.PAYMENT_FAILED;
        }
        return PaymentState.PAYMENT_PENDING;
    }

}
