package com.phonepe.sdk.javasdk;

import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.Endpoint;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import com.phonepe.sdk.javasdk.config.models.HystrixConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.SDKConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.PhonePeHttpClientFactory;
import com.phonepe.sdk.javasdk.transaction.checksum.ChecksumGeneratorFactory;
import com.phonepe.sdk.javasdk.transaction.checksum.payload.ChecksumGenerator;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiatorFactory;
import com.phonepe.sdk.javasdk.transaction.init.models.AllowedAccountConstraint;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.init.models.enums.CallbackMode;
import com.phonepe.sdk.javasdk.transaction.init.models.enums.RedirectMode;
import com.phonepe.sdk.javasdk.transaction.models.InitParams;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreator;
import com.phonepe.sdk.javasdk.transaction.request.PhonePeHttpRequestCreatorFactory;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;
import org.apache.commons.lang3.ObjectUtils;

import java.util.List;
import java.util.Set;

/**
 * Facade that wraps the {@link com.phonepe.sdk.javasdk.transaction.init.TransactionInitiator} and
 * {@link com.phonepe.sdk.javasdk.transaction.callback.CallbackHandler} and
 * {@link com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker} for the
 * user.
 */
@Slf4j
public class PhonePeClient {

    private TransactionInitiator transactionInitiator;

    private CallbackHandler callbackHandler;

    private TransactionStatusChecker statusChecker;

    private int defaultKeyIndex;

    @Builder
    public PhonePeClient(final SDKConfig sdkConfig) {
        Preconditions.checkNotNull(sdkConfig);
        init(sdkConfig);
    }

    private void init(final SDKConfig sdkConfig) {
        HttpClientConfig httpClientConfig = sdkConfig.getHttpClientConfig();
        Endpoint endpoint = sdkConfig.getEndpoint();
        StatusConfig statusConfig = sdkConfig.getStatusConfig();
        MerchantConfig merchantConfig = sdkConfig.getMerchantConfig();
        InitConfig initConfig = sdkConfig.getInitConfig();
        HystrixConfig hystrixConfig = sdkConfig.getHystrixConfig();
        ChecksumGeneratorFactory checksumGeneratorFactory = new ChecksumGeneratorFactory();
        OkHttpClient client = PhonePeHttpClientFactory.buildOkHttpClient(httpClientConfig,endpoint.getEnvironment().isSecured());
        TransactionClient transactionClient = new TransactionClient(endpoint,client, hystrixConfig);
        ChecksumGenerator statusChecksumGenerator = statusConfig.getApiVersion().accept(checksumGeneratorFactory);
        ChecksumGenerator initChecksumGenerator = initConfig.getApiVersion().accept(checksumGeneratorFactory);

        PhonePeHttpRequestCreatorFactory<InitRequest> phonePeHttpRequestCreatorFactory = new PhonePeHttpRequestCreatorFactory<InitRequest>();
        PhonePeHttpRequestCreator<InitRequest> phonePeHttpRequestCreator = initConfig.getApiVersion().accept(phonePeHttpRequestCreatorFactory);


        TransactionInitiatorFactory transactionInitiatorFactory = TransactionInitiatorFactory.builder()
                                                                                             .merchantConfig(merchantConfig)
                                                                                             .initConfig(initConfig)
                                                                                             .transactionClient(transactionClient)
                                                                                             .checksumGenerator(initChecksumGenerator)
                                                                                             .phonePeHttpRequestCreator(phonePeHttpRequestCreator)
                                                                                             .build();
        this.statusChecker  =    TransactionStatusChecker.builder()
                                                         .merchantConfig(merchantConfig)
                                                         .statusConfig(statusConfig)
                                                         .transactionClient(transactionClient)
                                                         .checksumGenerator(statusChecksumGenerator)
                                                         .build();

        this.callbackHandler =  CallbackHandler.builder()
                                               .merchantConfig(merchantConfig)
                                               .transactionStatusChecker(this.statusChecker)
                                               .build();

        this.transactionInitiator = initConfig.getInitType().accept(transactionInitiatorFactory);
        this.defaultKeyIndex = KeyUtils.getDefaultKeyIndex(merchantConfig.getApiKeys());
    }

    public InitResponse initTransaction(final InitParams initParam) throws PhonePeClientException{
        return initTransaction(initParam, defaultKeyIndex);
    }

    public InitResponse initTransaction(final InitParams initParams, final int keyIndex) throws PhonePeClientException{
        return this.transactionInitiator.initiateTransaction(initParams.getTransactionId(), initParams.getAmount(),
                                                             ObjectUtils.isNotEmpty(initParams.getOrderId()) ? initParams.getOrderId()
                                                                                                             : initParams.getTransactionId(),
                                                             initParams.getUserId(), initParams.getSubMerchant(),
                                                             initParams.getSubMerchantId(), initParams.getMobileNumber(),
                                                             initParams.getEmailId(), initParams.getShortName(),
                                                             initParams.getMessage(), initParams.getOfferTags(),
                                                             initParams.getAllowedAccountConstraints(), initParams.getRedirectMode(),
                                                             initParams.getRedirectURL(), initParams.getCallbackMode(),
                                                             initParams.getCallbackURL(), keyIndex);
    }

    public StatusResponse handleCallback(final String responseReceived, final String checksum,
                                        final long transactionAmount) throws PhonePeClientException{
        return this.callbackHandler.handleCallback(responseReceived, checksum, transactionAmount);
    }

    public StatusResponse checkTransactionStatus(final String transactionId) throws PhonePeClientException{
        return checkTransactionStatus(transactionId,defaultKeyIndex);
    }

    public StatusResponse checkTransactionStatus(final String transactionId, final int keyIndex) throws PhonePeClientException{
        return this.statusChecker.checkTransactionStatus(transactionId, keyIndex);
    }

}