package com.phonepe.sdk.javasdk.http.utils;

import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import lombok.extern.slf4j.Slf4j;
import okhttp3.*;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * Utils class containing helper methods for Http.
 */
@Slf4j
public final class HttpUtils {

    public static final String APPLICATION_JSON = "application/json";
    public static final String CONTENT_TYPE = "Content-Type";
    public static final String CHECKSUM_HEADER = "X-VERIFY";
    public static final String LOCATION_HEADER = "location";
    public static final String PROVIDER_ID_HEADER = "X-PROVIDER-ID";
    public static final String REDIRECT_URL_HEADER = "X-REDIRECT-URL";
    public static final String REDIRECT_MODE_HEADER = "X-REDIRECT-MODE";
    public static final String CALLBACK_URL_HEADER = "X-CALLBACK-URL";
    public static final String CALL_MODE_HEADER = "X-CALL-MODE";
    public static final String V1_DEBIT_URL = "/v1/debit";
    public static final String V3_DEBIT_URL = "/v3/debit";


    private HttpUtils() {
        throw new IllegalStateException("Utility class");
    }


    public static byte[] body(final Response response) throws IOException {
        final ResponseBody body = response.body();
        if(ObjectUtils.isNotEmpty(body))
            return body.bytes();
        return new byte[]{};
    }

    public static HttpHeaderPair getHeaderPair(final String name, final String value){
        return HttpHeaderPair.builder()
                             .name(name)
                             .value(value)
                             .build();
    }

    public static HttpHeaderPair getContentTypeHeaderPair(){
        return getHeaderPair(CONTENT_TYPE, APPLICATION_JSON);
    }

    public static <T1, T2> void checkNotNull(T1 t, T2 t2, String message) throws PhonePeClientException{
        if (t == null && t2 == null) {
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR,message);
        }
    }

    public static List<HttpHeaderPair> getHttpHeaders(final String checksum,
                                                      final String providerId,
                                                      final String redirectURL,
                                                      final String redirectMode,
                                                      final String callbackURL,
                                                      final String callMode){
        Preconditions.checkNotNull(checksum);
        List<HttpHeaderPair> headerPairs = getHttpHeaders(checksum);
        if(StringUtils.isNotEmpty(providerId)){
            HttpHeaderPair providerIdHeader = getHeaderPair(PROVIDER_ID_HEADER, providerId);
            headerPairs.add(providerIdHeader);
        }
        if(StringUtils.isNotEmpty(redirectURL)){
            HttpHeaderPair redirectURLHeader = getHeaderPair(REDIRECT_URL_HEADER, redirectURL);
            headerPairs.add(redirectURLHeader);
        }
        if(StringUtils.isNotEmpty(redirectMode)){
            HttpHeaderPair redirectModeHeader = getHeaderPair(REDIRECT_MODE_HEADER, redirectMode);
            headerPairs.add(redirectModeHeader);
        }
        if(StringUtils.isNotEmpty(callbackURL)){
            HttpHeaderPair callbackURLHeader = getHeaderPair(CALLBACK_URL_HEADER, callbackURL);
            headerPairs.add(callbackURLHeader);
        }
        if(StringUtils.isNotEmpty(callMode)){
            HttpHeaderPair callbackModeHeader = getHeaderPair(CALL_MODE_HEADER, callMode);
            headerPairs.add(callbackModeHeader);
        }
        return headerPairs;
    }

    public static List<HttpHeaderPair> getHttpHeaders(final String checksum){
        List<HttpHeaderPair> headerPairs = new ArrayList<HttpHeaderPair>();
        HttpHeaderPair httpHeaderPair = getHeaderPair(CHECKSUM_HEADER, checksum);
        headerPairs.add(httpHeaderPair);
        HttpHeaderPair contentTypeHeaderPair = getContentTypeHeaderPair();
        headerPairs.add(contentTypeHeaderPair);
        return headerPairs;
    }

}

