package com.phonepe.sdk.javasdk.transaction.client;


import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.phonepe.sdk.javasdk.config.models.Endpoint;
import com.phonepe.sdk.javasdk.config.models.HttpClientConfig;
import com.phonepe.sdk.javasdk.config.models.HystrixConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.Consumer;
import com.phonepe.sdk.javasdk.http.HttpGetCommand;
import com.phonepe.sdk.javasdk.http.HttpPostCommand;
import com.phonepe.sdk.javasdk.http.models.ExtractedResponse;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpRequest;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import com.phonepe.sdk.javasdk.utils.MapperUtils;
import lombok.extern.slf4j.Slf4j;
import okhttp3.OkHttpClient;
import org.slf4j.Logger;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
public class TransactionClient {

    private final ObjectMapper objectMapper;
    private final OkHttpClient client;
    private final Endpoint endpoint;
    private final HystrixConfig hystrixConfig;

    public TransactionClient(final HttpClientConfig httpClientConfig,
                             final OkHttpClient okHttpClient,
                             final HystrixConfig hystrixConfig) {
        this.objectMapper = MapperUtils.getMapper();
        this.endpoint = httpClientConfig.getEndpoint();
        this.client = okHttpClient;
        this.hystrixConfig = hystrixConfig;
    }

    public <T> InitResponse initTransaction(final PhonePeHttpRequest<T> initRequest,
                                            final List<HttpHeaderPair> httpHeaders,
                                            final String url) throws PhonePeClientException {
        try {

            Consumer<ExtractedResponse,PhonePeHttpResponse<InitResponse>> initResponseConsumer = handleNonSuccessServiceCall(log,"initTransaction");
            PhonePeHttpResponse<InitResponse> phonePeHttpResponse = HttpPostCommand.<PhonePeHttpResponse<InitResponse>>builder()
                    .url(url)
                    .client(this.client)
                    .mapper(this.objectMapper)
                    .endpoint(this.endpoint)
                    .command("transactionInit")
                    .responseTypeReference(new TypeReference<PhonePeHttpResponse<InitResponse>>() {})
                    .headers(httpHeaders)
                    .httpRequest(initRequest)
                    .nonSuccessResponseConsumer(initResponseConsumer)
                    .hystrixConfig(this.hystrixConfig)
                    .build()
                    .executeTracked(TransactionClient.class);

            return phonePeHttpResponse.getData();
        } catch (PhonePeClientException e){
            log.error("Error while making call to PhonePe init transaction API");
            throw e;
        } catch (Exception e) {
            log.error("Error in calling PhonePe init transaction API", e);
            Map<String, Object> objectMap = new HashMap<String, Object>();
            objectMap.put("MESSAGE", e.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR,
                                             "Error executing http client: for initiating transaction"
                                             + ": " + e.getMessage(), objectMap, e);
        }
    }


    public PhonePeHttpResponse<TransactionStatusResponse> getTransactionStatus(final List<HttpHeaderPair> httpHeaders,
                                                                               final String url) throws PhonePeClientException{
        try {
            Consumer<ExtractedResponse,PhonePeHttpResponse<TransactionStatusResponse>> transactionResponseConsumer = handleNonSuccessServiceCall(log,"transactionStatus");
            return HttpGetCommand.<PhonePeHttpResponse<TransactionStatusResponse>>builder()
                    .url(url)
                    .client(this.client)
                    .mapper(this.objectMapper)
                    .endpoint(this.endpoint)
                    .command("transactionStatus")
                    .responseTypeReference(new TypeReference<PhonePeHttpResponse<TransactionStatusResponse>>() {})
                    .headers(httpHeaders)
                    .nonSuccessResponseConsumer(transactionResponseConsumer)
                    .hystrixConfig(this.hystrixConfig)
                    .build()
                    .executeTracked(TransactionClient.class);
        } catch (PhonePeClientException e){
            log.error("Error while making call to PhonePe transaction status API");
            throw e;
        } catch (Exception e) {
            log.error("Error in calling PhonePe transaction status API", e);
            Map<String, Object> objectMap = new HashMap<String, Object>();
            objectMap.put("MESSAGE", e.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR,
                                                    "Error executing http client: for transaction status"
                                                    + ": " + e.getMessage(), objectMap, e);
        }
    }

    public static <T> Consumer<ExtractedResponse, T> handleNonSuccessServiceCall(final Logger log,
                                                                                 final String serviceName) {
        return new Consumer<ExtractedResponse, T>() {
            @Override
            public T consume(ExtractedResponse extractedResponse) throws Exception {
                final int code = extractedResponse.getCode();
                final byte[] body = extractedResponse.getBody();
                log.error("Error while calling {} service code:{}, response:{}", serviceName, code, new String(body));
                throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_SERVER_ERROR, serviceName + "http call returned code"+ code);
            }
        };
    }
}
