package com.phonepe.sdk.javasdk.transaction.callback;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.exception.PhonePeClientException;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.transaction.status.TransactionStatusChecker;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import com.phonepe.sdk.javasdk.utils.Base64Utils;
import com.phonepe.sdk.javasdk.utils.ChecksumUtils;
import com.phonepe.sdk.javasdk.utils.KeyUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;

import java.util.HashMap;
import java.util.Map;

@Slf4j
public class CallbackHandler {

    private TransactionStatusChecker transactionStatusChecker;

    private MerchantConfig merchantConfig;

    @Builder
    public CallbackHandler(final TransactionStatusChecker transactionStatusChecker,
                           final MerchantConfig merchantConfig) {
        this.transactionStatusChecker = transactionStatusChecker;
        this.merchantConfig = merchantConfig;
    }

    public StatusResponse handleCallback(final String responseReceived, final String checksum, final long transactionAmount) throws PhonePeClientException {
        Preconditions.checkNotNull(responseReceived);
        Preconditions.checkNotNull(checksum);
        try{
            final String responseBody = Base64Utils.decodeBase64(responseReceived);
            final PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse = new ObjectMapper().readValue(responseBody, new
                    TypeReference<PhonePeHttpResponse<TransactionStatusResponse>>() {});
            final int keyIndex = KeyUtils.getKeyIndexFromChecksum(checksum);
            final TransactionStatusResponse callbackResponseData = transactionStatusResponse.getData();
            final String merchantId = callbackResponseData.getMerchantId();
            final Long callbackAmount = callbackResponseData.getAmount();
            if(!isValidTransactionAmount(callbackAmount, transactionAmount)){
                throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR, "Transaction amount passed does not match callback status response amount");
            }
            if(!isValidChecksum(responseReceived, checksum, keyIndex) || !isValidMerchantId(merchantId)){
                throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR, "Invalid checksum or merchantId in callback response");
            }
            final String transactionId = transactionStatusResponse.getData().getTransactionId();
            StatusResponse statusResponse =  this.transactionStatusChecker.checkTransactionStatus(transactionId, keyIndex);
            if(!isValidTransactionAmount(callbackAmount, statusResponse.getAmount())){
                throw new PhonePeClientException(PhonePeClientException.ErrorCode.VALIDATION_ERROR, "Transaction amount mismatch in callback response and status check response");
            }
            return statusResponse;
        } catch (PhonePeClientException e){
            log.error("Exception while handling callback: {}", e.getMessage());
            throw e;
        } catch (Exception ex){
            log.error("Error in calling PhonePe init transaction API", ex);
            Map<String, Object> objectMap = new HashMap<String, Object>();
            objectMap.put("MESSAGE", ex.getMessage());
            throw new PhonePeClientException(PhonePeClientException.ErrorCode.HTTP_CLIENT_ERROR,
                                             "Error executing http client: for initiating transaction"
                                             + ": " + ex.getMessage(), objectMap, ex);
        }
    }

    private boolean isValidMerchantId(final String merchantId){
        return this.merchantConfig.getMerchantId().equals(merchantId);
    }

    private boolean isValidChecksum(final String responseBody, final String checksum, final int keyIndex) throws Exception{
        final String apiKey = KeyUtils.getAPIKeyFromIndex(this.merchantConfig.getApiKeys(),keyIndex);
        final String generatedChecksum = ChecksumUtils.generateChecksumValue(responseBody, apiKey);
        return checksum.equalsIgnoreCase(generatedChecksum);
    }

    private boolean isValidTransactionAmount(final Long callbackAmount, final Long statusResponseAmount){
        return callbackAmount.equals(statusResponseAmount);
    }

}