package com.phonepe.sdk.javasdk.transaction.init;

import com.phonepe.sdk.javasdk.config.models.APIKeyConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.http.PhonePeException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.init.models.AllowedAccountConstraint;
import com.phonepe.sdk.javasdk.transaction.models.Base64Request;
import com.phonepe.sdk.javasdk.utils.Base64Utils;
import com.phonepe.sdk.javasdk.utils.ChecksumUtils;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * Template Method class for initiating the transaction for any given type of flow.
 */
@Slf4j
public abstract class TransactionInitiator {

    private TransactionClient transactionClient;

    private InitConfig initConfig;

    private MerchantConfig merchantConfig;

    public TransactionInitiator(final TransactionClient transactionClient,
                                final InitConfig initConfig,
                                final MerchantConfig merchantConfig) {
        this.transactionClient = transactionClient;
        this.initConfig = initConfig;
        this.merchantConfig = merchantConfig;
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final Long amount,
                                            final String merchantOrderId,
                                            final String merchantUserId,
                                            final String subMerchant,
                                            final String subMerchantId,
                                            final String mobileNumber,
                                            final String emailId,
                                            final String shortName,
                                            final Set<String> offerTags,
                                            final List<AllowedAccountConstraint> allowedAccountConstraints,
                                            final String redirectMode,
                                            final String redirectURL,
                                            final String callbackMode,
                                            final String callbackURL,
                                            final int keyIndex) throws PhonePeException{
        TransactionInitComponent transactionInitComponent = new BaseTransactionInit();
        if (CollectionUtils.isNotEmpty(offerTags)) {
            transactionInitComponent = new OfferInitDecorator(transactionInitComponent, offerTags);
        }
        if (CollectionUtils.isNotEmpty(allowedAccountConstraints)) {
            transactionInitComponent = new AccountConstraintsDecorator(transactionInitComponent, allowedAccountConstraints);
        }
        final InitRequest initRequest = transactionInitComponent.getInitRequest(this.merchantConfig.getMerchantId(), transactionId,amount);
        return initiateTransaction(initRequest, redirectMode, redirectURL, callbackMode, callbackURL, keyIndex);
    }

    private InitResponse initiateTransaction(final InitRequest initRequest,
                                             final String redirectMode,
                                             final String redirectURL,
                                             final String callMode,
                                             final String callbackURL,
                                             final int keyIndex) throws PhonePeException{

        try{

            final Base64Request base64Request = Base64Utils.getBase64Request(initRequest);
            final String apiURL = getAPIURL();
            final String apiKey = getAPIKeyFromIndex(keyIndex);
            final String checksumBody = ChecksumUtils.generateChecksumBody(base64Request.getRequest(), apiURL, apiKey);
            final String checksumHeader = ChecksumUtils.generateChecksumHeader(checksumBody, keyIndex);
            final List<HttpHeaderPair> httpHeaders = getHttpHeaders(checksumHeader,
                                                                    this.merchantConfig.getProviderId(),
                                                                    redirectURL,
                                                                    redirectMode,
                                                                    callbackURL,
                                                                    callMode);
            return performTransactionInit(base64Request, httpHeaders, apiURL);
        } catch (Exception ex){
            log.error("Exception occurred in while initiating transactions");
            throw new PhonePeException(ex.getMessage(), ex);
        }
    }

    protected abstract InitRequest buildInitRequest(final String merchantId,
                                                    final String transactionId,
                                                    final String userId,
                                                    final Long amount,
                                                    final String orderId);

    private InitResponse performTransactionInit(final Base64Request base64Request,
                                                final List<HttpHeaderPair> httpHeaders,
                                                final String apiURL) throws PhonePeException{
        try{
            return this.transactionClient.initTransaction(base64Request,httpHeaders,apiURL);
        } catch (Exception ex){
            log.error("Exception occurred");
            throw new PhonePeException("Could not perform transaction Initiation");
        }
    }

    private String getAPIURL(){
        return String.format("/%s/debit", this.initConfig.getApiVersion().getApiVersion());
    }

    private List<HttpHeaderPair> getHttpHeaders(final String checksumHeader,
                                                final String providerId,
                                                final String redirectURL,
                                                final String redirectMode,
                                                final String callbackURL,
                                                final String callMode){
        List<HttpHeaderPair> headerPairs = new ArrayList<HttpHeaderPair>();
        HttpHeaderPair checksumHeaderPair = HttpUtils.getHeaderPair("X-VERIFY", checksumHeader);
        headerPairs.add(checksumHeaderPair);
        HttpHeaderPair contentTypeHeaderPair = HttpUtils.getContentTypeHeaderPair();
        headerPairs.add(contentTypeHeaderPair);
        if(StringUtils.isNotEmpty(providerId)){
            HttpHeaderPair providerIdHeader = HttpUtils.getHeaderPair("X-PROVIDER-ID", providerId);
            headerPairs.add(providerIdHeader);
        }
        if(StringUtils.isNotEmpty(redirectURL)){
            HttpHeaderPair redirectURLHeader = HttpUtils.getHeaderPair("X-REDIRECT-URL", redirectURL);
            headerPairs.add(redirectURLHeader);
        }
        if(StringUtils.isNotEmpty(redirectMode)){
            HttpHeaderPair redirectModeHeader = HttpUtils.getHeaderPair("X-REDIRECT-MODE", redirectMode);
            headerPairs.add(redirectModeHeader);
        }
        if(StringUtils.isNotEmpty(callbackURL)){
            HttpHeaderPair callbackURLHeader = HttpUtils.getHeaderPair("X-CALLBACK-URL", callbackURL);
            headerPairs.add(callbackURLHeader);
        }
        if(StringUtils.isNotEmpty(callMode)){
            HttpHeaderPair callbackModeHeader = HttpUtils.getHeaderPair("X-CALL-MODE", callMode);
            headerPairs.add(callbackModeHeader);
        }
        return headerPairs;
    }

    private String getAPIKeyFromIndex(int keyIndex) throws Exception{
        for(APIKeyConfig apiKey : this.merchantConfig.getApiKeys()){
            if(apiKey.getKeyIndex() == keyIndex)
                return apiKey.getKeyValue();
        }
        throw new PhonePeException();
    }

}
