package com.phonepe.sdk.javasdk.config;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.TreeTraversingParser;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.PhonePeSDKConfig;
import lombok.Builder;
import okhttp3.HttpUrl;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;

import javax.validation.ConstraintViolation;
import javax.validation.Validation;
import javax.validation.Validator;

public class ConfigurationFactory<T> {

  private final Class<T> clazz;
  private final ObjectMapper objectMapper;
  private final Validator validator;
  private final JsonFactory parserFactory;
  private static final String SDK_CONFIG_FILE = "phonepe_java_sdk_properties.yml";

  @Builder
  public ConfigurationFactory(Class<T> clazz, ObjectMapper objectMapper) {
    this.clazz = clazz;
    this.parserFactory = new YAMLFactory();
    this.validator =  Validation.buildDefaultValidatorFactory()
                                .getValidator();
    this.objectMapper = objectMapper;
    //this.objectMapper.findAndRegisterModules();
    this.objectMapper.enable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
  }

  @Builder
  public ConfigurationFactory(Class<T> clazz) {
    this(clazz, new ObjectMapper());
  }

  public T build() throws IOException, ConfigurationException {
    return build(SDK_CONFIG_FILE);
  }

  //TODO: Ensure that file can be taken from anywhere in the classpath
  public T build(String fileName) throws IOException, ConfigurationException {
    Preconditions.checkNotNull(fileName);
    File file = null;
    try {
      file = new File(fileName);
      InputStream inputStream = getClass().getClassLoader().getResourceAsStream(SDK_CONFIG_FILE);
      final JsonNode node = objectMapper.readTree(createParser(inputStream));
      final String filename = file.toString();
      return build(node, filename);
    } catch (JsonParseException exception) {
      throw new ConfigurationException(file.toString(), exception.getMessage());
    }
  }

  protected JsonParser createParser(InputStream inputStream) throws IOException {
    return parserFactory.createParser(inputStream);
  }

  private T build(JsonNode node, String filename) throws IOException, ConfigurationException {
    final T config = objectMapper.readValue(new TreeTraversingParser(node), clazz);
    validate(filename, config);
    return config;
  }

  private void validate(String file, T config) throws ConfigurationException {
    if (validator != null) {
      final Set<ConstraintViolation<T>> violations = validator.validate(config);
      if (!violations.isEmpty()) {
        throw new ConfigurationException(file, violations);
      }
    }
  }
}