package com.phonepe.sdk.javasdk.transaction.status;

import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.Retryer;
import com.google.common.base.Predicate;
import com.phonepe.sdk.javasdk.config.models.APIKeyConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.config.models.StatusConfig;
import com.phonepe.sdk.javasdk.http.PhonePeException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.models.PhonePeHttpResponse;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.transaction.status.models.StatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.TransactionStatusResponse;
import com.phonepe.sdk.javasdk.transaction.status.models.enums.PaymentState;
import com.phonepe.sdk.javasdk.utils.ChecksumUtils;
import com.phonepe.sdk.javasdk.utils.RetryUtils;
import lombok.Builder;
import lombok.extern.slf4j.Slf4j;

import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.Callable;

@Slf4j
public class TransactionStatusChecker {


    private TransactionClient transactionClient;

    private StatusConfig statusConfig;

    private MerchantConfig merchantConfig;

    @Builder
    public TransactionStatusChecker(final TransactionClient transactionClient,
                                    final MerchantConfig merchantConfig,
                                    final StatusConfig statusConfig) {
        this.transactionClient = transactionClient;
        this.merchantConfig = merchantConfig;
        this.statusConfig = statusConfig;
    }

    public StatusResponse checkTransactionStatus(final String transactionId){
        return checkTransactionStatus(transactionId,1);
    }

    public StatusResponse checkTransactionStatus(final String transactionId,
                                                 final int keyIndex){
        StatusResponse statusResponse = null;
        try{
            final String apiURL = getAPIURL(this.merchantConfig.getMerchantId(),transactionId);
            final String apiKey = getAPIKeyFromIndex(keyIndex);
            final String checksumBody = ChecksumUtils.generateChecksumBody("",
                                                                           apiURL,
                                                                           apiKey);
            final String checksumHeader = ChecksumUtils.generateChecksumHeader(checksumBody,
                                                                               keyIndex);
            List<HttpHeaderPair> httpHeaders = getHttpHeaders(checksumHeader);
            Callable<PhonePeHttpResponse<TransactionStatusResponse>> transactionStatusCallable = getCallable(apiURL,
                                                                                                             httpHeaders);
            Predicate<PhonePeHttpResponse<TransactionStatusResponse>> phonePeHttpResponsePredicate = getRetryPredicate();
            Retryer<PhonePeHttpResponse<TransactionStatusResponse>> retryer = RetryUtils.getRetryer(statusConfig.getRetryConfig(),
                                                                                                    phonePeHttpResponsePredicate);
            PhonePeHttpResponse<TransactionStatusResponse> phonePeHttpResponse = retryer.call(transactionStatusCallable);
            statusResponse = buildStatusResponse(phonePeHttpResponse);
        } catch (RetryException ex){
            PhonePeHttpResponse<TransactionStatusResponse> phonePeHttpResponse = (PhonePeHttpResponse<TransactionStatusResponse>) ex
                    .getLastFailedAttempt()
                    .getResult();
            statusResponse = buildStatusResponse(phonePeHttpResponse);
        }
        catch (Exception ex){
            log.error("Exception occurred while performing transaction status check");
        }
        return statusResponse;
    }

    private String getAPIURL(final String merchantId, final String transactionId){
        return String.format("/%s/transaction/%s/%s/status",
                             this.statusConfig.getApiVersion().getApiVersion(),
                             merchantId,
                             transactionId);
    }


    private List<HttpHeaderPair> getHttpHeaders(final String checksumHeader){
        List<HttpHeaderPair> headerPairs = new ArrayList<HttpHeaderPair>();
        HttpHeaderPair httpHeaderPair = HttpUtils.getHeaderPair("X-VERIFY", checksumHeader);
        HttpHeaderPair contentTypeHeaderPair = HttpUtils.getContentTypeHeaderPair();
        headerPairs.add(contentTypeHeaderPair);
        return Collections.singletonList(httpHeaderPair);
    }

    //TODO: Check how the duplication of this method along with
    //TODO: the same method in transaction initiator can be eliminated.
    private String getAPIKeyFromIndex(int keyIndex) throws Exception{
        for(APIKeyConfig apiKey : this.merchantConfig.getApiKeys()){
            if(apiKey.getKeyIndex() == keyIndex)
                return apiKey.getKeyValue();
        }
        throw new PhonePeException();
    }

    private Callable<PhonePeHttpResponse<TransactionStatusResponse>> getCallable(final String apiURL,
                                                                                 final List<HttpHeaderPair> headerPairs){
        return  new Callable<PhonePeHttpResponse<TransactionStatusResponse>>() {
            @Override
            public PhonePeHttpResponse<TransactionStatusResponse> call() throws Exception {
                return transactionClient.getTransactionStatus(headerPairs, apiURL);
            }
        };
    }

    private Predicate<PhonePeHttpResponse<TransactionStatusResponse>> getRetryPredicate(){
        return new Predicate<PhonePeHttpResponse<TransactionStatusResponse>>(){
            @Override
            public boolean apply(@Nullable PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse) {
                return transactionStatusResponse.getCode().equalsIgnoreCase("PAYMENT_PENDING") ||
                       transactionStatusResponse.getCode().equalsIgnoreCase("INTERNAL_SERVER_ERROR");
            }
        };
    }

    private StatusResponse buildStatusResponse(final PhonePeHttpResponse<TransactionStatusResponse> transactionStatusResponse){
        return StatusResponse.builder()
                             .transactionId(transactionStatusResponse.getData().getTransactionId())
                             .paymentState(getPaymentState(transactionStatusResponse.getCode()))
                             .providerReferenceId(transactionStatusResponse.getData().getProviderReferenceId())
                             .amount(transactionStatusResponse.getData().getAmount())
                             .build();
    }

    private PaymentState getPaymentState(String statusCode){
        if("PAYMENT_SUCCESS".equalsIgnoreCase(statusCode)){
            return PaymentState.PAYMENT_SUCCESS;
        }
        if("PAYMENT_ERROR".equalsIgnoreCase(statusCode) ||
           "PAYMENT_DECLINED".equalsIgnoreCase(statusCode) ||
           "PAYMENT_CANCELLED".equalsIgnoreCase(statusCode) ||
           "TRANSACTION_NOT_FOUND".equalsIgnoreCase(statusCode)){
            return PaymentState.PAYMENT_FAILED;
        }
        return PaymentState.PAYMENT_PENDING;
    }

}
