package com.phonepe.sdk.javasdk.transaction.init;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.phonepe.sdk.javasdk.config.models.APIKeyConfig;
import com.phonepe.sdk.javasdk.config.models.InitConfig;
import com.phonepe.sdk.javasdk.config.models.MerchantConfig;
import com.phonepe.sdk.javasdk.http.PhonePeException;
import com.phonepe.sdk.javasdk.http.models.HttpHeaderPair;
import com.phonepe.sdk.javasdk.http.utils.HttpUtils;
import com.phonepe.sdk.javasdk.transaction.client.TransactionClient;
import com.phonepe.sdk.javasdk.utils.ChecksumUtils;
import com.phonepe.sdk.javasdk.transaction.init.models.InitRequest;
import com.phonepe.sdk.javasdk.transaction.init.models.InitResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * Template Method class for initiating the transaction for any given type of flow.
 */
@Slf4j
public abstract class TransactionInitiator {

    private TransactionClient transactionClient;

    private InitConfig initConfig;

    private MerchantConfig merchantConfig;

    public TransactionInitiator(final TransactionClient transactionClient,
                                final InitConfig initConfig,
                                final MerchantConfig merchantConfig) {
        this.transactionClient = transactionClient;
        this.initConfig = initConfig;
        this.merchantConfig = merchantConfig;
    }


    public InitResponse initiateTransaction(final String transactionId,
                                            final String userId,
                                            final Long amount){
        return initiateTransaction(transactionId, "", "", "", "", 1,userId,amount,transactionId);
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final int keyIndex,
                                            final String userId,
                                            final Long amount){
        return initiateTransaction(transactionId, "", "", "", "", keyIndex,userId,amount,transactionId);
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final int keyIndex,
                                            final String userId,
                                            final Long amount,
                                            final String orderId){
        return initiateTransaction(transactionId, "", "", "", "", keyIndex,userId,amount,orderId);
    }

    public InitResponse initiateTransaction(final String transactionId,
                                            final String redirectMode,
                                            final String redirectURL,
                                            final String callbackURL,
                                            final String callMode,
                                            final int keyIndex,
                                            final String userId,
                                            final Long amount,
                                            final String orderId){

        Preconditions.checkNotNull(transactionId);
        Preconditions.checkNotNull(orderId);
        Preconditions.checkNotNull(amount);
        InitResponse initResponse = null;
        try{
            InitRequest initRequest = buildInitRequest(this.merchantConfig.getMerchantId(),
                                                       transactionId,
                                                       userId,
                                                       amount,
                                                       orderId);
            // TODO: Move this into a separate utils class.
            final String jsonBody = new ObjectMapper().writeValueAsString(initRequest);
            final String b64RequestBody = ChecksumUtils.encodeBase64(jsonBody);
            final String apiURL = getAPIURL();
            final String apiKey = getAPIKeyFromIndex(keyIndex);
            final String checksumBody = ChecksumUtils.generateChecksumBody(b64RequestBody, apiURL, apiKey);
            final String checksumHeader = ChecksumUtils.generateChecksumHeader(checksumBody, keyIndex);
            List<HttpHeaderPair> httpHeaders = getHttpHeaders(checksumHeader,
                                                              this.merchantConfig.getProviderId(),
                                                              redirectURL,
                                                              redirectMode,
                                                              callbackURL,
                                                              callMode);
            initResponse =  performTransactionInit(initRequest,
                                                   httpHeaders,
                                                   apiURL);
        } catch (Exception ex){
            log.error("Exception occurred");
        }
        return initResponse;
    }

    protected abstract InitRequest buildInitRequest(final String merchantId,
                                                    final String transactionId,
                                                    final String userId,
                                                    final Long amount,
                                                    final String orderId);

    private InitResponse performTransactionInit(final InitRequest initRequest,
                                                final List<HttpHeaderPair> httpHeaders,
                                                final String apiURL) throws Exception{
        try{
            return this.transactionClient.initTransaction(initRequest,httpHeaders,apiURL);
        } catch (Exception ex){
            throw new Exception();
        }
    }

    private String getAPIURL(){
        return String.format("/%s/debit",
                             this.initConfig.getApiVersion().getApiVersion());
    }

    private List<HttpHeaderPair> getHttpHeaders(final String checksumHeader,
                                                final String providerId,
                                                final String redirectURL,
                                                final String redirectMode,
                                                final String callbackURL,
                                                final String callMode){
        List<HttpHeaderPair> headerPairs = new ArrayList<HttpHeaderPair>();
        HttpHeaderPair checksumHeaderPair = HttpUtils.getHeaderPair("X-VERIFY", checksumHeader);
        headerPairs.add(checksumHeaderPair);
        HttpHeaderPair contentTypeHeaderPair = HttpUtils.getContentTypeHeaderPair();
        headerPairs.add(contentTypeHeaderPair);
        if(StringUtils.isNotEmpty(providerId)){
            HttpHeaderPair providerIdHeader = HttpUtils.getHeaderPair("X-PROVIDER-ID", providerId);
            headerPairs.add(providerIdHeader);
        }
        if(StringUtils.isNotEmpty(redirectURL)){
            HttpHeaderPair redirectURLHeader = HttpUtils.getHeaderPair("X-REDIRECT-URL", redirectURL);
            headerPairs.add(redirectURLHeader);
        }
        if(StringUtils.isNotEmpty(redirectMode)){
            HttpHeaderPair redirectModeHeader = HttpUtils.getHeaderPair("X-REDIRECT-MODE", redirectMode);
            headerPairs.add(redirectModeHeader);
        }
        if(StringUtils.isNotEmpty(callbackURL)){
            HttpHeaderPair callbackURLHeader = HttpUtils.getHeaderPair("X-CALLBACK-URL", callbackURL);
            headerPairs.add(callbackURLHeader);
        }
        if(StringUtils.isNotEmpty(callMode)){
            HttpHeaderPair callbackModeHeader = HttpUtils.getHeaderPair("X-CALL-MODE", callMode);
            headerPairs.add(callbackModeHeader);
        }
        return headerPairs;
    }

    private String getAPIKeyFromIndex(int keyIndex) throws Exception{
        for(APIKeyConfig apiKey : this.merchantConfig.getApiKeys()){
            if(apiKey.getKeyIndex() == keyIndex)
                return apiKey.getKeyValue();
        }
        throw new PhonePeException();
    }


}