package com.phonepe.intent.sdk.widgets;

import android.graphics.Bitmap;
import android.net.Uri;
import android.net.http.HttpResponseCache;
import android.os.Build;
import android.util.Log;
import android.webkit.WebResourceRequest;
import android.webkit.WebResourceResponse;
import android.webkit.WebView;
import android.webkit.WebViewClient;

import com.phonepe.intent.sdk.api.PhonePe;
import com.phonepe.intent.sdk.api.PhonePeInitException;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.TLSSocketFactory;
import com.phonepe.intent.sdk.utils.Utils;

import java.net.URL;
import java.util.List;

import javax.net.ssl.HttpsURLConnection;

/**
 * @author Sharath Pandeshwar
 * @since 28/04/17.
 */

public class PhWebViewClient extends WebViewClient {

    private static final String TAG = "WEB_VIEW_CLIENT";

    public static final String HTTPS = "https";

    private List<String> mWhiteListedPrefixes;

    private static boolean usePreCache;
    private static List<String> approvedFileTypes;

    public void setWhiteListedUrls(List<String> urls) {
        mWhiteListedPrefixes = urls;
    }

    public static void setUsePreCache(boolean usePreCache) {
        PhWebViewClient.usePreCache = usePreCache;
    }

    public static void setApprovedFileTypes(List<String> approvedFileTypes) {
        PhWebViewClient.approvedFileTypes = approvedFileTypes;
    }

    @Override
    public void onPageStarted(WebView view, String url, Bitmap favicon) {
        if (!isSecure(url)) {
            view.destroy();
            SdkLogger.w(TAG, String.format("view is destroyed for url = {%s}", url));
        }
        super.onPageStarted(view, url, favicon);
    }

    private boolean isSecure(String url) {
        if (mWhiteListedPrefixes == null || mWhiteListedPrefixes.isEmpty()) {
            return true;
        }
        mWhiteListedPrefixes.add(HTTPS);
        for (String whiteListedUrl : mWhiteListedPrefixes) {
            if (url.startsWith(whiteListedUrl)) {
                return true;
            }
        }

        SdkLogger.w(TAG, String.format("url = {%s} is not secure", url));
        return false;
    }

    @Override
    public WebResourceResponse shouldInterceptRequest(WebView view, WebResourceRequest request) {

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {

            SdkLogger.d(TAG, "request can not be intercepted, function itself added in LOLLIPOP. Callback not expected here. calling super..");
            return super.shouldInterceptRequest(view, request);
        }

        if(!usePreCache) {
            SdkLogger.i(TAG, "Precache has been disabled either by merchant or config");
            return super.shouldInterceptRequest(view, request);
        }

        //ignore api check already done in start
        Uri tempUrl = request.getUrl();
        ObjectFactory objectFactory = null;
        try {
            objectFactory = PhonePe.getObjectFactory();
        } catch (PhonePeInitException e) {
            SdkLogger.e(TAG, e.getMessage(), e);
            return super.shouldInterceptRequest(view, request);
        }
        if (view == null
                || request == null
                || Utils.setUpResponseCache(objectFactory) == false
                || tempUrl == null
                || request.getMethod().matches("POST")) {


            SdkLogger.d(TAG, String.format("request check 1 = {url:%s, method:%s} is not a valid candidate for interception", tempUrl, request.getMethod()));
            return super.shouldInterceptRequest(view, request);
        }

        String path = tempUrl.getPath();
        boolean match=false;
        for(int i=0; i<approvedFileTypes.size(); i++) {
            if(path.endsWith(approvedFileTypes.get(i))) {
                match = true;
                continue;
            }
        }

        if(!match) {
            SdkLogger.d(TAG, String.format("request check 2 = {url:%s, method:%s} is not a valid candidate for interception", tempUrl, request.getMethod()));
            return super.shouldInterceptRequest(view, request);
        }

        try {

            URL url = new URL(tempUrl.toString());
            HttpsURLConnection httpsURLConnection = (HttpsURLConnection) url.openConnection();

            // Setting the socket factory only above API level 21
            TLSSocketFactory tlsSocketFactory = new TLSSocketFactory();
            httpsURLConnection.setSSLSocketFactory(tlsSocketFactory);

            httpsURLConnection.setRequestMethod(request.getMethod());
            httpsURLConnection.setUseCaches(true);
            httpsURLConnection.connect();
            int responseCode = httpsURLConnection.getResponseCode();
            if(responseCode >= 200 && responseCode < 400) {
                final String content_type = httpsURLConnection.getContentType();
                final String separator = "; charset=";
                if (content_type != null) {
                    int pos = -1;
                    pos = content_type.indexOf(separator);
                    final String mime_type = pos >= 0 ? content_type.substring(0, pos) : content_type;
                    final String encoding = pos >= 0 ? content_type.substring(pos + separator.length()) : "UTF-8";
                    SdkLogger.d(TAG, url.toString());
                    return new WebResourceResponse(mime_type, encoding, httpsURLConnection.getInputStream());
                }
                SdkLogger.e(TAG, String.format("failed to intercept request because of either content_type is null or invalid. content_type = {%s}",
                        content_type));
            }
            return super.shouldInterceptRequest(view, request);
        } catch (Throwable e) {

            SdkLogger.e(TAG, String.format("failed to intercept the request, because of throwable = {%s}", e.getMessage()));
            return super.shouldInterceptRequest(view, request);
        }
    }

}
