package com.phonepe.intent.sdk.utils;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;

import com.phonepe.intent.sdk.contracts.iDeviceIdListener;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;

/**
 * @author TheEternalWitness
 * @since 10/04/18.
 */
public class DeviceInfoProvider implements ObjectFactoryInitializationStrategy {

    private Context context;
    private DeviceIdGenerator deviceIdGenerator;
    private ObjectFactory objectFactory;

    protected ObjectFactory getObjectFactory() {

        return this.objectFactory;
    }

    protected Context getApplicationContext() {
        if(context == null) {
            this.context = objectFactory.getApplicationContext();
        }
        return this.context;
    }

    protected DeviceIdGenerator getDeviceIdGenerator() {
        return this.deviceIdGenerator;
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        this.context = objectFactory.getApplicationContext();
        this.deviceIdGenerator = objectFactory.<DeviceIdGenerator>get(DeviceIdGenerator.class);
        this.objectFactory = objectFactory;
    }

    @Override
    public boolean isCachingAllowed() {
        return true;
    }

    /**
     * @return the current version of the application
     */
    public String getAppVersion() {
        String appVersion = "unknown";
        try {
            PackageInfo pInfo = getApplicationContext().getPackageManager().getPackageInfo(getApplicationContext().getPackageName(), 0);
            appVersion = pInfo.versionName;

        } catch (Exception e) {
            //Do nothing
        }

        return appVersion;
    }

    public String getOsVersion() {
        return String.valueOf(Build.VERSION.SDK_INT);
    }

    /**
     * @return Model of device
     */
    public String getModel() {
        return Build.MODEL;
    }

    /**
     * @return Manufacturer of device
     */
    public String getManufacturer() {
        return Build.MANUFACTURER;
    }

    /**
     * @return package name of the application
     */
    public String getPackageName() {
        String appId = null;
        try {
            PackageInfo pInfo = getApplicationContext().getPackageManager().getPackageInfo(getApplicationContext().getPackageName(), 0);
            appId = pInfo.packageName;

        } catch (Exception e) {
            //Do nothing
        }

        return appId;
    }

    public String getAppId() {
        String appId = "";
        try {
            ApplicationInfo applicationInfo = getApplicationContext().getPackageManager().getApplicationInfo(getApplicationContext().getPackageName(), PackageManager.GET_META_DATA);
            if (applicationInfo != null && applicationInfo.metaData != null) {
                appId = applicationInfo.metaData.getString("com.phonepe.android.sdk.AppId");
            }
        } catch (Exception e) {
            // throw new IllegalArgumentException("Failed to retrieve MerchantId MetaData from manifest.");
        }

        return appId;
    }

    public void getDeviceFingerPrint(iDeviceIdListener listener) {
        deviceIdGenerator.generateDeviceId(getApplicationContext(), listener);
    }

    public void getDeviceFingerPrintForUpi(final iDeviceIdListener listener) {
        getDeviceFingerPrint(new iDeviceIdListener() {

            @Override
            public void onDeviceIdAvailable(String deviceId) {
                if (listener != null) {
                    listener.onDeviceIdAvailable(getUpiDeviceIdFromDeviceId(deviceId));
                }
            }
        });
    }

    public String getUpiDeviceIdFromDeviceId(String deviceId) {
        if (deviceId == null) {
            return null;
        }

        /**
         * NOTE: DO NOT CHANGE THIS LOGIC, as its in sync with server
         */
        String phoneId = (deviceId.length() > 35) ? deviceId.replaceAll("-", "").toUpperCase() : deviceId;
        phoneId = phoneId.substring(0, Math.min(35, phoneId.length()));
        return phoneId;
    }

}
