package com.phonepe.intent.sdk.utils;

import android.annotation.SuppressLint;
import android.content.ContentResolver;
import android.content.Context;
import android.os.AsyncTask;
import android.provider.Settings;
import android.support.annotation.WorkerThread;
import android.util.Base64;
import android.util.Log;
import android.util.Pair;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.phonepe.intent.sdk.contracts.iDeviceIdListener;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;

import java.security.SecureRandom;
import java.util.Date;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;


/**
 * Generator for Device Id, Ideally this device id should be unique for any
 * particular device, and it should not change until unless something has changes genuinely.
 */
public class DeviceIdGenerator implements ObjectFactoryInitializationStrategy {

    public static final String TAG = "MerchantTransaction";
    static volatile String mDeviceFingerPrint;
    private boolean encrypt = false;
    private ObjectFactory objectFactory;
    private final Executor executor = new ThreadPoolExecutor(1, 100, 1000, TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(10));

    //*********************************************************************
    // APIs
    //*********************************************************************

    public void generateDeviceId(Context context, iDeviceIdListener listener) {
        generateDeviceId(context, false, listener);
    }

    public void generateDeviceId(Context context, boolean encrypted, iDeviceIdListener listener) {
        encrypt = encrypted;
        if (Utils.isBelowAndroidOreo()) {
            generateDeviceIdBelowO(context.getContentResolver(), listener);
        } else {
            generateDeviceIdForOOrAbove(context, listener);
        }
    }


    //*********************************************************************
    // Private utility methods
    //*********************************************************************

    private void generateDeviceIdBelowO(ContentResolver contentResolver, iDeviceIdListener listener) {
        String androidId = Settings.Secure.getString(contentResolver, Settings.Secure.ANDROID_ID);
        String deviceFingerPrint = BullsEye.fingerprint(androidId);
        SdkLogger.d(TAG, "MerchantTransaction ID via earlier mechanism as " + deviceFingerPrint);

        if (listener != null) {
            sendResponse(listener, deviceFingerPrint);
        }
    }

    /**
     * Will return generated or pre-cached deviceFingerprint
     *
     * @return deviceFingerprint
     */
    @SuppressLint("StaticFieldLeak")
    private void generateDeviceIdForOOrAbove(final Context context, final iDeviceIdListener listener) {
        if (mDeviceFingerPrint != null) {
            sendResponse(listener, mDeviceFingerPrint);
        } else {
            synchronized (DeviceIdGenerator.class) {
                new AsyncTask<Void, Void, String>() {
                    @Override
                    protected String doInBackground(Void... voids) {
                        String fingerprint;
                        try {
                            String adId = getAdId(context);
                            if (adId != null) {
                                fingerprint = BullsEye.fingerprint(adId);
                            } else {
                                String androidId = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
                                fingerprint = BullsEye.fingerprint(androidId);
                            }
                            return fingerprint;
                        } catch (Exception e) {
                            SdkLogger.e(TAG, e.getMessage(), e);
                            return null;
                        }
                    }

                    @Override
                    protected void onPostExecute(String fingerprint) {
                        super.onPostExecute(fingerprint);
                        if (fingerprint != null) {
                            mDeviceFingerPrint = fingerprint;
                            SdkLogger.d(TAG, "MerchantTransaction ID via new mechanism as " + mDeviceFingerPrint);
                        } else {
                            String androidId = Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID);
                            mDeviceFingerPrint = BullsEye.fingerprint(androidId);
                            SdkLogger.d(TAG, "MerchantTransaction ID via old mechanism as ad Id was not available:" + mDeviceFingerPrint);
                        }

                        if (listener != null) {
                            sendResponse(listener, mDeviceFingerPrint);
                        }
                    }
                }.executeOnExecutor(executor);
            }
        }
    }

    /**
     * Return google advertisement Id.
     *
     * @param context
     * @return
     */
    @WorkerThread
    private String getAdId(Context context) {
        try {
            AdvertisingIdClient.Info adInfo;
            adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
            if (adInfo != null) {
                return adInfo.getId();
            }
            return null;
        } catch (Exception e) {
            SdkLogger.e(TAG, String.format("Exception caught while getting AdId, exception message = {%s}", e.getMessage()), e);
        }

        return null;
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {
        this.objectFactory = objectFactory;
    }

    @Override
    public boolean isCachingAllowed() {
        return true;
    }

    private String encryptFingerprint(String fingerprint) {
        String signature = DeviceIdGenerator.this.objectFactory.getPackageSignature();
        CryptLib cryptLib = new CryptLib();
        try {
            byte[] hash;
            byte[] ivBytes;
//
//            StringBuffer hashtext = new StringBuffer();
//            for (int i = 0; i < hash.length; i++) {
//                String hex = Integer.toHexString(0xff & hash[i]);
//                if (hex.length() == 1) hashtext.append('0');
//                hashtext.append(hex);
//            }
//            SdkLogger.d(TAG, "Message: " + fingerprint + " Password: " + hash + " Hashtext: " + hashtext);
            Date date = new Date();
            Pair<byte[], byte[]> encryptedData = encrypt((fingerprint+":"+date.getTime()).getBytes(), cryptLib.SHA256(signature));
            hash = encryptedData.first;//cryptLib.encrypt((fingerprint+":"+date.getTime()).getBytes(), cryptLib.SHA256(signature));
            ivBytes = encryptedData.second;
            StringBuffer encryptedText = new StringBuffer();
            for (int i = 0; i < hash.length; i++) {
                String hex = Integer.toHexString(0xff & hash[i]);
                if (hex.length() == 1) encryptedText.append('0');
                encryptedText.append(hex);
            }
            StringBuffer encryptedIv = new StringBuffer();
            for (int i = 0; i < ivBytes.length; i++) {
                String hex = Integer.toHexString(0xff & ivBytes[i]);
                if (hex.length() == 1) encryptedIv.append('0');
                encryptedIv.append(hex);
            }
            // Uncomment for debugging purpose
//            SdkLogger.d(TAG, "ID: " + encryptedText);
//            byte[] decryptKey = cryptLib.SHA256(signature);
//
//            byte[] decryptedmessage = cryptLib.decrypt(hash, decryptKey);
//            StringBuffer decryptedBuffer = new StringBuffer();
//            for (int i = 0; i < decryptedmessage.length; i++) {
//                String hex = Integer.toHexString(0xff & decryptedmessage[i]);
//                if (hex.length() == 1) decryptedBuffer.append('0');
//                decryptedBuffer.append(hex);
//            }
//            SdkLogger.d(TAG, "Decrypted: " + decryptedBuffer.toString());
            String deviceFingerprint = encryptedText.toString();
            String iv = encryptedIv.toString();
            String appId = objectFactory.get(Constants.MerchantMeta.APP_ID);
            return Base64.encodeToString((iv+deviceFingerprint+":"+appId).getBytes("UTF-8"), Base64.NO_WRAP);
        } catch (Exception e) {
            return null;
        }
    }

    public static Pair<byte[], byte[]> encrypt(byte[] value, byte[] secret) {
        try {
            SecretKeySpec secretKey = new SecretKeySpec(secret, "AES");

            // generate random IV
            SecureRandom secureRandom = new SecureRandom();
            byte[] iv = new byte[16];
            secureRandom.nextBytes(iv);
            Log.d("MainActivity", "encrypt: "+iv);
            IvParameterSpec ivspec = new IvParameterSpec(iv);

            //Encrypt using AES-CTR
            Cipher cipher = Cipher.getInstance("AES/CTR/NoPadding");
            cipher.init(Cipher.ENCRYPT_MODE, secretKey, ivspec);
            byte[] encyptedBytes = cipher.doFinal(value);

            return new Pair<>(encyptedBytes, iv);
        } catch (Exception e) {
            SdkLogger.e( TAG,"Error in encrypting value", e);
            return null;
        }
    }

    private void sendResponse(iDeviceIdListener listener, String fingerprint) {
        listener.onDeviceIdAvailable(encrypt ? encryptFingerprint(fingerprint) : fingerprint);
    }
    //*********************************************************************
    // Private classes/interfaces
    //*********************************************************************

    //*********************************************************************
    // End of the class
    //*********************************************************************
}
