package com.phonepe.intent.sdk.utils;

import android.support.annotation.NonNull;
import android.util.Base64;

import com.phonepe.intent.sdk.BuildConfig;
import com.phonepe.intent.sdk.core.BaseConfig;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;
import com.phonepe.intent.sdk.models.Event;
import com.phonepe.intent.sdk.models.EventsPayload;
import com.phonepe.intent.sdk.models.SDKContext;
import com.phonepe.intent.sdk.networking.APIHelper;
import com.phonepe.intent.sdk.networking.APIManager;
import com.phonepe.intent.sdk.networking.INetworkResponseListener;
import com.phonepe.intent.sdk.networking.NetworkConstants;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Map;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.EVENT_BODY_REQUEST;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.TRANSACTION_ID;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.X_AUTH_TOKEN;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.BACK_PRESSED_EVENT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.PHONEPE_APP_OPENED_FOR_RESULT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.PHONEPE_APP_RETURNED_RESULT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_BACK_BUTTON_CLICKED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_BACK_CANCELLED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_BACK_CONFIRMED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_BASE_WEB_ACTIVITY_CREATED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_CACHE_METRICS;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_DEFAULT_EVENT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_ERROR_TO_USER;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_INITIALIZATION_FAILED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_LAUNCHED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_MERCHANT_CALLBACK_SENT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_NETWORK_ERROR;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_PAGE_LOAD_COMPLETE;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_RENDER_START;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_RUNTIME_ERROR;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_TRANSACTION_REQUEST_CREATED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_TRANSACTION_TOKEN_RECEIVED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_UPI_APP_SELECTION_ACTIVITY_STARTED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_UPI_APP_STARTED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_WEB_VIEW_CONSOLE_ERROR;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_AVAILABILITY_CHECK_REQUEST_CREATED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.CHECK_AVAILABILITY_CALL_SUCCESS;

public class AnalyticsManager implements ObjectFactoryInitializationStrategy {

    public interface Events {
        String SDK_DEFAULT_EVENT = "DEFAULT_EVENT";
        String SDK_INITIALIZATION_FAILED = "SDK_INITIALIZATION_FAILED";
        String SDK_TRANSACTION_REQUEST_CREATED = "SDK_TRANSACTION_REQUEST_CREATED";
        String SDK_DEBIT_SUGGEST_REQUEST_CREATED = "SDK_DEBIT_SUGGEST_REQUEST_CREATED";
        String SDK_GENERATE_DEEPLINK_REQUEST_CREATED = "SDK_GENERATE_DEEPLINK_REQUEST_CREATED";
        String SDK_BASE_WEB_ACTIVITY_CREATED = "SDK_BASE_WEB_ACTIVITY_CREATED";
        String SDK_WEB_VIEW_CONSOLE_ERROR = "SDK_WEB_VIEW_CONSOLE_ERROR";
        String SDK_NETWORK_ERROR = "SDK_NETWORK_ERROR";
        String SDK_LAUNCHED = "SDK_LAUNCHED";
        String SDK_ERROR_TO_USER = "SDK_ERROR_TO_USER";
        String SDK_TRANSACTION_TOKEN_RECEIVED = "SDK_TRANSACTION_TOKEN_RECEIVED";
        String SDK_PAGE_LOAD_COMPLETE = "SDK_PAGE_LOAD_COMPLETE";
        String SDK_BACK_CONFIRMED = "SDK_BACK_CONFIRMED";
        String SDK_BACK_CANCELLED = "SDK_BACK_CANCELLED";
        String SDK_MERCHANT_CALLBACK_SENT = "SDK_MERCHANT_CALLBACK_SENT";
        String SDK_BACK_BUTTON_CLICKED = "SDK_BACK_BUTTON_CLICKED";
        String BACK_PRESSED_EVENT = "BACK_PRESSED";
        String SDK_RUNTIME_ERROR = "RUNTIME_ERROR";
        String SDK_RENDER_START = "SDK_RENDER_START";
        String SDK_CACHE_METRICS = "SDK_PRE_CACHE_METRICS";
        String SDK_UPI_APP_SELECTION_ACTIVITY_STARTED = "SDK_UPI_APP_SELECTION_ACTIVITY_STARTED";
        String SDK_UPI_APP_STARTED = "SDK_UPI_APP_STARTED";
        String PHONEPE_APP_OPENED_FOR_RESULT = "PHONEPE_APP_OPENED_FOR_RESULT";
        String PHONEPE_APP_RETURNED_RESULT = "PHONEPE_APP_RETURNED_RESULT";
        String SDK_AVAILABILITY_CHECK_REQUEST_CREATED = "SDK_AVAILABILITY_CHECK_REQUEST_CREATED";
        String CHECK_AVAILABILITY_CALL_SUCCESS = "CHECK_AVAILABILITY_CALL_SUCCESS";
    }

    public interface EventConstants {
        String X_AUTH_TOKEN = "X-AUTH-TOKEN";
        String EVENT_TIME = "eventTime";
        String INTENT_SUPPORTED = "intentSupported";
        String INTENT_URL = "intentUrl";
        String RESULT = "result";
        String WAS_CANCELED = "wasCanceled";
        String ACTION = "action";
        String BACK_PRESS_EVENT_INFO = "back press";
        String SDK_TRANSACTION_STATUS = "sdkTransactionStatus";
        String KEY_EVENT_NAME = "eventName";
        String KEY_EVENT_ID = "eventId";
        String KEY_DATA = "data";
        String SESSION_ID = "X-SDK-SESSION-ID";
        String ERROR_MESSAGE = "errorMessage";
        String SDK_FLOW_TYPE = "sdkFlowType";
        String APP_NAME = "upiAppName";
        String TRANSACTION_ID = "transactionId";
        String EVENT_BODY_REQUEST = "request";
    }


    public static final String TAG = "AnalyticsManager";
    private ObjectFactory objectFactory;
    private APIHelper apiHelper;
    private APIManager apiManager;
    private Event emptyEvent;
    private String headerToken;
    private AnalyticsManagerConfig analyticsManagerConfig;

    public void setHeaderToken(@NonNull String headerToken) {
        this.headerToken = headerToken;
        SdkLogger.i(TAG, "header token received, trying to submit cached events ...");
        submitPersistedEvents(headerToken);
    }

    private void submitPersistedEvents(String headerToken) {

        String eventsPayload = this.analyticsManagerConfig.getPersistedEvents();
        if (eventsPayload == null) {

            SdkLogger.i(TAG, "event database is empty");
            return;
        }
        this.analyticsManagerConfig.clearAllData();
        String[] eventsPayloadArray = eventsPayload.split(AnalyticsManagerConfig.EVENT_LIST_SEPARATOR);
        SdkLogger.d(TAG, String.format("persisted {%d} events in previous sessions are fetched", eventsPayloadArray.length));
        for (String payload : eventsPayloadArray) {
            sendEvents(headerToken, payload);
        }
    }

    private void sendEvents(String token, final String jsonArrayString) {

        Map<String, String> headers = this.objectFactory.<String, String>getHashMap();
        if (token != null) {
            headers.put(X_AUTH_TOKEN, token);
        }

        try {
            final EventsPayload eventsPayload = objectFactory.<EventsPayload>get(EventsPayload.class);
            eventsPayload.setEvents(jsonArrayString);
            eventsPayload.setSdkContext(objectFactory.get(SDKContext.class));
            eventsPayload.setMerchantId(objectFactory.<String>get(Constants.MerchantMeta.MERCHANT_ID));
            eventsPayload.setTransactionId(objectFactory.<String>get(TRANSACTION_ID));
            String eventsPayloadString = eventsPayload.toJsonString().replace("\\/", "/").replace("\n", "");
            String base64Payload = Base64.encodeToString(eventsPayloadString.getBytes("UTF-8"), Base64.NO_WRAP);
            String checkSum = getCheckSum(base64Payload);
            final JSONObject payload = new JSONObject();
            payload.put(EVENT_BODY_REQUEST, base64Payload);
            SdkLogger.d(TAG, eventsPayload.toJsonString());
            headers.put(Constants.GenericConstants.SDK_CHECKSUM, checkSum);
            boolean isInUATMode = Utils.isTrue(this.objectFactory.<Boolean>get(Constants.MerchantMeta.IS_UAT));
            final String endpoint = NetworkConstants.getApiBaseUrl(isInUATMode) + NetworkConstants.APIEndPoints.API_EVENT_INJESTION;
            this.apiHelper.getCommonHeadersAdded(headers, new APIHelper.OnHeadersAddedListener() {
                @Override
                public void onHeadersAdded(Map<String, String> addedHeaders) {
                    AnalyticsManager.this.apiManager.asyncPostRequest(endpoint, addedHeaders, payload.toString(), new INetworkResponseListener() {
                        @Override
                        public void onSuccess(String response) {
                            SdkLogger.v(TAG, String.format("events!!! ingestion succeeded. count = {%d}, payload = {%s}", objectFactory.getJsonArray(jsonArrayString).length(),  eventsPayload.toJsonString()));
                        }

                        @Override
                        public void onFailure(int responseCode, String response) {
                            SdkLogger.e(TAG, String.format("Failed to inject events!!!, count = {%d},responseCode = {%d}, response = {%s} for payload = {%s}", objectFactory.getJsonArray(jsonArrayString).length(), responseCode, response, jsonArrayString));
                            // AnalyticsManager.this.analyticsManagerConfig.saveEvents(jsonArrayString);

                        }
                    }, false);
                }
            });
        } catch (Exception e) {
            SdkLogger.e(TAG, e.getMessage(), e);
        }

    }

    private void sendEvent(Event event) {
        final JSONArray jsonArray = this.objectFactory.getJsonArray();
         jsonArray.put(event.toJsonObject());
        if (jsonArray.length() != 0) {
            sendEvents(this.headerToken, jsonArray.toString());
        }
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        this.objectFactory = objectFactory;
        this.emptyEvent = getEvent(SDK_DEFAULT_EVENT);
        this.analyticsManagerConfig = this.objectFactory.<AnalyticsManagerConfig>get(AnalyticsManagerConfig.class);
        this.apiHelper = this.objectFactory.<APIHelper>get(APIHelper.class);
        this.apiManager = this.objectFactory.<APIManager>get(APIManager.class);
    }

    @Override
    public boolean isCachingAllowed() {
        return true;
    }


    public void submit(Event event) {

        if (! Constants.GenericConstants.RELEASE.equalsIgnoreCase(BuildConfig.BUILD_TYPE)) {
            String eventName = event.getEventName();
            switch (eventName) {
                case SDK_LAUNCHED:
                    break;
                case PHONEPE_APP_OPENED_FOR_RESULT:
                    break;
                case SDK_TRANSACTION_TOKEN_RECEIVED:
                    break;
                case PHONEPE_APP_RETURNED_RESULT:
                    break;
                case SDK_PAGE_LOAD_COMPLETE:
                    break;
                case SDK_BACK_BUTTON_CLICKED:
                    break;
                case SDK_BACK_CONFIRMED:
                    break;
                case SDK_BACK_CANCELLED:
                    break;
                case SDK_MERCHANT_CALLBACK_SENT:
                    break;
                case BACK_PRESSED_EVENT:
                    break;
                case SDK_RENDER_START:
                    break;
                case SDK_CACHE_METRICS:
                    break;

                // below are new events added
                case SDK_INITIALIZATION_FAILED:
                case SDK_RUNTIME_ERROR:
                case SDK_TRANSACTION_REQUEST_CREATED:
                case SDK_BASE_WEB_ACTIVITY_CREATED:
                case SDK_WEB_VIEW_CONSOLE_ERROR:
                case SDK_ERROR_TO_USER:
                case SDK_NETWORK_ERROR:
                case SDK_UPI_APP_SELECTION_ACTIVITY_STARTED:
                case SDK_AVAILABILITY_CHECK_REQUEST_CREATED:
                case CHECK_AVAILABILITY_CALL_SUCCESS:
                case SDK_UPI_APP_STARTED:
                  break;
                default:
                    event = this.emptyEvent;
                    break;
            }


            if (event.equals(this.emptyEvent)) {

                RuntimeExceptionManager runtimeExceptionManager = this.objectFactory.getRuntimeExceptionManager();
                runtimeExceptionManager.submit(TAG, String.format("trying to submit empty event for eventName = {%s}", eventName), RuntimeExceptionManager.Severity.LOW);
            }
        }

        sendEvent(event);
    }


    public Event getEvent(String eventName) {

        SdkLogger.v(TAG, String.format("preparing event with name : {%s}", eventName));
        Event event = this.objectFactory.<Event>get(Event.class);
        event.setEventName(eventName);
        return event;
    }

    private String getCheckSum(String base64Payload) {
        try {
            String endPoint = NetworkConstants.APIEndPoints.API_EVENT_INJESTION;
//            String signature = this.objectFactory.getPackageSignature();
            String signature = "YHyczmD0QO9iCWDbF9bm6T3Ysy0=";
            CryptLib cryptLib = new CryptLib();
            byte[] hash = cryptLib.SHA256(base64Payload+endPoint+signature);
            StringBuffer hashtext = new StringBuffer();

            for (int i = 0; i < hash.length; i++) {
                String hex = Integer.toHexString(0xff & hash[i]);
                if (hex.length() == 1) hashtext.append('0');
                hashtext.append(hex);
            }
            // Uncomment this when you want to debug Checksum logic
            SdkLogger.d(TAG, "Base64Payload: "+base64Payload+"\n : "+endPoint+"\nSignature: "+signature+"\nHashText:"+hashtext);
            return hashtext.toString();
        } catch (Exception e) {
            SdkLogger.e(TAG, e.getMessage(), e);
            return null;
        }
    }

    public static class AnalyticsManagerConfig extends BaseConfig {

        private static final String ANALYTICS_MANAGER_PREF_FILE = "1bca992e-c98b-4ab7-a789-737ec20fd436";
        private static final String EVENT_LIST = "1bca992e";
        private static final String EVENT_LIST_SEPARATOR = "@%#";

        @Override
        protected String getPreferenceName() {
            return ANALYTICS_MANAGER_PREF_FILE;
        }

        public String getPersistedEvents() {

            return getString(EVENT_LIST, null);
        }

        public void saveEvents(String jsonArrayString) {

            String jsonArrayStringInFile = getPersistedEvents();

            if (jsonArrayStringInFile != null) {

                SdkLogger.i(TAG, "events for previous sessions are available in db");
                jsonArrayString = jsonArrayString + EVENT_LIST_SEPARATOR + jsonArrayStringInFile;
                SdkLogger.v(TAG, String.format("event json string = {%s} .", jsonArrayString));
            }

            SdkLogger.i(TAG, "saving events in local db ...");
            saveString(EVENT_LIST, jsonArrayString);
        }
    }

}
