package com.phonepe.intent.sdk.ui;

import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.v7.app.AlertDialog;
import android.transition.Fade;
import android.view.View;
import android.view.Window;
import android.webkit.ConsoleMessage;
import android.webkit.WebChromeClient;
import android.webkit.WebView;
import android.widget.TextView;

import com.phonepe.intent.sdk.R;
import com.phonepe.intent.sdk.api.TransactionRequest;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.models.IntentResponse;
import com.phonepe.intent.sdk.models.SDKConfig;
import com.phonepe.intent.sdk.models.SDKContext;
import com.phonepe.intent.sdk.models.Response;
import com.phonepe.intent.sdk.models.UrlConfig;
import com.phonepe.intent.sdk.mvp.TransactionPresenter;
import com.phonepe.intent.sdk.mvp.iTransactionPresenter;
import com.phonepe.intent.sdk.mvp.iTransactionView;
import com.phonepe.intent.sdk.utils.AnalyticsManager;
import com.phonepe.intent.sdk.utils.Config;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;
import com.phonepe.intent.sdk.widgets.PhWebViewClient;

import java.util.List;

//import in.juspay.juspaysafe.BrowserCallback;
//import in.juspay.juspaysafe.BrowserParams;
//import in.juspay.juspaysafe.JuspaySafeBrowser;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.ERROR_MESSAGE;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_LAUNCHED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_WEB_VIEW_CONSOLE_ERROR;

/**
 * @author TheEternalWitness
 * @since 31/03/18.
 */
public class TransactionActivity extends BaseWebActivity implements iTransactionView {

    public static final String TAG = "TransactionActivity";
    public static final String TRX_VIEW = "trxView";
    public static final String HEADER_HOLDER = "headerHolder";
    public static final String BRIDGE_CALLBACK = "bridgeCallback";
    public static final int PP_REQUEST_CODE = 725;
    private iTransactionPresenter transactionPresenter;
    private TextView retryButton;
    private TransactionWebClient transactionWebClient;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            getWindow().getAttributes().windowAnimations = R.style.FadeAnimation;
        }
        SdkLogger.d(TAG, "transaction activity creating...");
        ObjectFactory objectFactory = null;

        if (getIntent() != null && getIntent().getExtras() != null) {

            objectFactory = getIntent().<ObjectFactory>getParcelableExtra(Constants.BundleConstants.DATA_FACTORY);
            ObjectFactory.InitializationBundle initializationBundle = objectFactory.<ObjectFactory.InitializationBundle>get(ObjectFactory.InitializationBundle.class);
            initializationBundle.put(TRX_VIEW, this);
            initializationBundle.put(BRIDGE_CALLBACK, this);
            this.transactionPresenter = objectFactory.<TransactionPresenter>get(TransactionPresenter.class, initializationBundle);

        } else if (savedInstanceState != null) {

            objectFactory = savedInstanceState.getParcelable(Constants.BundleConstants.DATA_FACTORY);
            ObjectFactory.InitializationBundle initializationBundle = objectFactory.<ObjectFactory.InitializationBundle>get(ObjectFactory.InitializationBundle.class);
            initializationBundle.put(TRX_VIEW, this);
            initializationBundle.put(BRIDGE_CALLBACK, this);
            this.transactionPresenter = objectFactory.<TransactionPresenter>get(TransactionPresenter.class, initializationBundle);
        }

        super.onCreate(savedInstanceState);
        initViews();
        getTransactionPresenter().onRequestReceived(getIntent(), savedInstanceState);
        SdkLogger.d(TAG, "transaction activity created.");
    }

    protected iTransactionPresenter getTransactionPresenter() {

        return this.transactionPresenter;
    }

    @Override
    public void onBackPressed() {
        if (getTransactionPresenter() != null) {
            getTransactionPresenter().onBackPressed();
        }
    }

    @Override
    public void onSaveInstanceState(Bundle outState) {
        super.onSaveInstanceState(outState);
        outState.putParcelable(Constants.BundleConstants.DATA_FACTORY, getObjectFactory());
        if (getTransactionPresenter() != null) {
            getTransactionPresenter().saveInstanceState(outState);
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (getTransactionPresenter() != null) {
            getTransactionPresenter().onDestroy();
        }
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        SdkLogger.d(TAG, String.format("activity result received. requestCode = {%s}, resultCode = {%s}, Intent = {%s}.", requestCode, resultCode, data == null ? "null" : data.toString()));
        if (requestCode == PP_REQUEST_CODE) {
            boolean userCancelled = resultCode == Activity.RESULT_CANCELED;
            IntentResponse intentresponse = IntentResponse.getResult(data);
            getTransactionPresenter().onActivityResult(userCancelled, intentresponse);
            Response failureResponse = getObjectFactory().getResponse(Constants.GenericConstants.FAILURE);
            String response = intentresponse != null ? intentresponse.toString() : failureResponse.toJsonString();
            handleResult(response, userCancelled);
        } else {
            super.onActivityResult(requestCode, resultCode, data);
        }
    }

    //*********************************************************************
    // Views related
    //*********************************************************************

    private void initViews() {

        SdkLogger.d(TAG, "initializing views..");
        SdkLogger.d(TAG, "views initialized");
    }

    @Override
    protected void initWebView() {
        SdkLogger.d(TAG, "initializing web views..");
        List<String> mAllowedPrefixes = getObjectFactory().<String>getArrayList();
        transactionWebClient = new TransactionWebClient(mAllowedPrefixes);
        getWebView().setWebViewClient(transactionWebClient);
        getWebView().setWebChromeClient(new TransactionWebChromeClient());
        super.initWebView();
        SdkLogger.d(TAG, "web views initialized");
    }

    //*********************************************************************
    // View interface implementations
    //*********************************************************************

    @Override
    public void openActivityForResult(Uri uri) {

        SdkLogger.d(TAG, String.format("request activity start for result for uri = {%s}.", uri));
        Intent intent = new Intent(Intent.ACTION_VIEW, uri);
        intent.setPackage(Utils.getPhonePePackageName(getObjectFactory()));
        if (!isFinishing()) {
            SdkLogger.d(TAG, String.format("starting activity for intent = {%s}.", intent.toString()));
            hideProgressBar();
            startActivityForResult(intent, PP_REQUEST_CODE);
        }

    }

    @Override
    public void endWithResult(String result) {
        handleResult(result, false);
    }

    @Override
    public void endWithCancel(String result) {
        handleResult(result, true);
    }

    @Override
    public Context getContext() {
        return this;
    }

    @Override
    public void setLoadingStatus(boolean status) {
        if (status) {
            runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    showLoading();
                }
            });
        } else {
            hideLoading();
        }
    }

    @Override
    public void showAlert() {
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setMessage(getString(R.string.cancel_confirmation))
                .setCancelable(false)
                .setPositiveButton(getString(R.string.ok), new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int id) {
                        onBackPressConfirmed();
                    }
                })
                .setNegativeButton(getString(R.string.cancel), new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int id) {
                        onBackPressCancelled();
                        dialog.cancel();
                    }
                });

        AlertDialog alert = builder.create();
        alert.show();
        alert.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(this.getResources().getColor(R.color.colorText));
        alert.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(this.getResources().getColor(R.color.colorText));
    }

    @Override
    public void showError(String errorMessage, boolean showRetry) {
        super.showError(errorMessage, showRetry);
        String errorStatusMessageFormat = getObjectFactory().<Config>get(Config.class).getErrorStatusMessageFormat(showRetry);
        String applicationName = Utils.getAppNameFor(getObjectFactory(), getPackageName());
        String statusMessage = String.format(errorStatusMessageFormat, errorMessage, applicationName);
        AlertDialog.Builder builder = new AlertDialog.Builder(this);
        builder.setMessage(statusMessage)
                .setCancelable(false);
        if(showRetry) {
            builder.setPositiveButton("Retry", new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                    dialog.dismiss();
                    getTransactionPresenter().onRetryPressed();
                }
            }).setNegativeButton("Close", new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                    dialog.dismiss();
                    getTransactionPresenter().onComplete(Constants.GenericConstants.FAILURE);
                }
            });
        }
        AlertDialog alertDialog = builder.create();
        alertDialog.show();
        alertDialog.getButton(AlertDialog.BUTTON_NEGATIVE).setTextColor(this.getResources().getColor(R.color.colorText));
        alertDialog.getButton(AlertDialog.BUTTON_POSITIVE).setTextColor(this.getResources().getColor(R.color.colorText));
        //retryButton.setVisibility(showRetry ? View.VISIBLE : View.GONE);
    }

    //*********************************************************************
    // Bridge related
    //*********************************************************************

    @Override
    public void onComplete(String val) {
        if (getTransactionPresenter() != null) {
            getTransactionPresenter().onComplete(val);
        }
    }

    @Override
    public void onJSLoadStateChanged(String context, String isJSLoaded, String callback) {
        getTransactionPresenter().onJSLoadStateChanged(context, isJSLoaded, callback);
    }

    @Override
    public void showLoader(String context, String showLoader, String callback) {
        getTransactionPresenter().onShowLoader(context, showLoader, callback);
    }

//    @Override
//    public void onJusPayTransactionCompleted(String transactionId, String status, boolean isSuccess) {
//        getTransactionPresenter().onJusPayTransactionCompleted(transactionId, status, isSuccess);
//    }
//
//    @Override
//    public void startJustPay(BrowserParams browserParams, BrowserCallback browserCallback, String[] endUrls) {
//        JuspaySafeBrowser.setEndUrls(endUrls);
//        JuspaySafeBrowser.start(getContext(), browserParams, browserCallback);
//    }

    @Override
    public void keepUrl(UrlConfig urlConfig) {
        getTransactionPresenter().keepUrl(urlConfig);
    }


    //*********************************************************************
    // Private classes
    //*********************************************************************

    private class TransactionWebClient extends PhWebViewClient {

        TransactionWebClient(List<String> mAllowedUrls) {
            super.setWhiteListedUrls(mAllowedUrls);
        }

        @Override
        public void onPageStarted(WebView view, String url, Bitmap favicon) {
            if (getTransactionPresenter() != null && url != null) {
                getTransactionPresenter().onPageLoadStart(url);
            }
        }

        @Override
        public void onPageFinished(WebView view, String url) {
            if (getTransactionPresenter() != null && url != null) {
                getTransactionPresenter().onPageFinished(url);
            }
        }
    }

    private class TransactionWebChromeClient extends WebChromeClient {
        @Override
        public boolean onConsoleMessage(ConsoleMessage consoleMessage) {
            SDKConfig sdkConfig = getObjectFactory().get(SDKConfig.class);
            if (sdkConfig.isWebLogsEnabled()) {
                ConsoleMessage.MessageLevel messageLevel = consoleMessage.messageLevel();
                switch (messageLevel) {
                    case LOG:
                        SdkLogger.webL("WebViewConsole", consoleMessage.message(), consoleMessage.lineNumber(), consoleMessage.sourceId());
                        break;
                    case ERROR:
                        SdkLogger.webE("WebViewConsole", consoleMessage.message(), consoleMessage.lineNumber(), consoleMessage.sourceId());
                        break;
                    case DEBUG:
                        SdkLogger.webD("WebViewConsole", consoleMessage.message(), consoleMessage.lineNumber(), consoleMessage.sourceId());
                        break;
                    case WARNING:
                        SdkLogger.webW("WebViewConsole", consoleMessage.message(), consoleMessage.lineNumber(), consoleMessage.sourceId());
                        break;
                    case TIP:
                        SdkLogger.webT("WebViewConsole", consoleMessage.message(), consoleMessage.lineNumber(), consoleMessage.sourceId());
                        break;
                }
            }
            AnalyticsManager analyticsManager = getObjectFactory().<AnalyticsManager>get(AnalyticsManager.class);
            analyticsManager.submit(analyticsManager.getEvent(SDK_WEB_VIEW_CONSOLE_ERROR).putInEventBody(ERROR_MESSAGE, consoleMessage.message()));
            return false;
        }

        @Override
        public void onProgressChanged(WebView view, int progress) {
            if("card".matches(getLoaderType())) {
                if (progress < 95) {
                    if (getWebView() != null && getWebView().getVisibility() == View.VISIBLE) {
                        SdkLogger.d("WEB_VIEW_FAIL", "hidding webview");
                        getWebView().setVisibility(View.INVISIBLE);
                    }
                } else if (progress >= 95) {
                    if (getWebView() != null && getWebView().getVisibility() != View.VISIBLE) {
                        SdkLogger.d("WEB_VIEW_FAIL", "showing webview");
                        getWebView().setVisibility(View.VISIBLE);
                        hideLoading();
                    }
                }
            }
        }
    }

    //*********************************************************************
    // Helper methods
    //*********************************************************************

    private void onBackPressConfirmed() {
        getTransactionPresenter().onBackPressConfirmed();
    }

    private void onBackPressCancelled() {
        getTransactionPresenter().onBackPressCancelled();
    }


    private void handleResult(String result, boolean isCanceled) {

        Intent intent = new Intent();
        intent.putExtra(Constants.GenericConstants.KEY_TRANSACTION_RESULT, result);
        if (isCanceled) {
            setResult(RESULT_CANCELED, intent);
        } else {
            setResult(RESULT_OK, intent);
        }

        finish();
    }

    public static Intent getIntent(Context context, TransactionRequest transactionRequest, ObjectFactory objectFactory) {

        SDKContext sdkContext = objectFactory.<SDKContext>get(SDKContext.class);
        Bundle bundle = objectFactory.getInitializedBundle();
        bundle.putParcelable(Constants.BundleConstants.KEY_REQUEST, transactionRequest);
        bundle.putParcelable(Constants.BundleConstants.KEY_SDK_CONTEXT, sdkContext);
        sendSDKLaunchedEvent(objectFactory);
        return objectFactory.createIntent(context, TransactionActivity.class, bundle);
    }

    public static Intent getImplicit(Context context, TransactionRequest transactionRequest, @NonNull ObjectFactory objectFactory) {

        SDKContext sdkContext = objectFactory.<SDKContext>get(SDKContext.class);
        Bundle bundle = objectFactory.getInitializedBundle();
        bundle.putParcelable(Constants.BundleConstants.KEY_REQUEST, transactionRequest);
        bundle.putParcelable(Constants.BundleConstants.KEY_SDK_CONTEXT, sdkContext);
        sendSDKLaunchedEvent(objectFactory);
        return objectFactory.createIntent(context, UpiAppsSelectionDialogActivity.class, bundle);
    }


    protected static void sendSDKLaunchedEvent(ObjectFactory objectFactory) {

        AnalyticsManager analyticsManager = objectFactory.<AnalyticsManager>get(AnalyticsManager.class);
        analyticsManager.submit(analyticsManager.getEvent(SDK_LAUNCHED));
    }

    @Override
    public void openUrlInWebView(String url) {
        SDKConfig sdkConfig = getObjectFactory().get(SDKConfig.class);
        TransactionWebClient.setUsePreCache(sdkConfig.shouldUsePreCache() && Utils.isTrue(getObjectFactory().<Boolean>get(Constants.MerchantMeta.PRE_CACHE_ENABLED)));
        TransactionWebClient.setApprovedFileTypes(sdkConfig.getApprovedFileTypes());
        super.openUrlInWebView(url);
    }

    //*********************************************************************
    // End of class
    //*********************************************************************
}
