package com.phonepe.intent.sdk.networking;

import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.support.annotation.NonNull;

import com.phonepe.intent.sdk.BuildConfig;
import com.phonepe.intent.sdk.api.AvailabilityCheckRequest;
import com.phonepe.intent.sdk.api.TransactionRequest;
import com.phonepe.intent.sdk.contracts.iDeviceIdListener;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;
import com.phonepe.intent.sdk.models.SDKContext;
import com.phonepe.intent.sdk.networking.models.AvailabilityCheckAPIRequest;
import com.phonepe.intent.sdk.networking.models.DebitInitRequest;
import com.phonepe.intent.sdk.networking.models.DeviceEventPayload;
import com.phonepe.intent.sdk.networking.models.PhonePeContext;
import com.phonepe.intent.sdk.networking.models.APIRequest;
import com.phonepe.intent.sdk.utils.AnalyticsManager;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.CryptLib;
import com.phonepe.intent.sdk.utils.DeviceInfoProvider;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

import java.util.Map;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.SESSION_ID;

/**
 * @author TheEternalWitness
 * @since 10/04/18.
 */
public class APIHelper implements ObjectFactoryInitializationStrategy {

    private ObjectFactory objectFactory;
    private APIManager apiManager;
    private DeviceInfoProvider deviceInfoProvider;
    private static final String TAG = "PhonePe";


    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        this.apiManager = objectFactory.<APIManager>get(APIManager.class);
        this.deviceInfoProvider = objectFactory.<DeviceInfoProvider>get(DeviceInfoProvider.class);
        this.objectFactory = objectFactory;
    }

    @Override
    public boolean isCachingAllowed() {
        return true;
    }

    protected APIManager getApiManager() {

        return this.apiManager;
    }

    protected DeviceInfoProvider getDeviceInfoProvider() {
        return this.deviceInfoProvider;
    }

    protected ObjectFactory getObjectFactory() {

        return this.objectFactory;
    }

    //*********************************************************************
    // Public APIs
    //*********************************************************************

    public void checkPhonePeAvailability(final AvailabilityCheckRequest availabilityCheckRequest, SDKContext sdkContext, PhonePeContext phonePeContext, @NonNull final INetworkResponseListener listener) {
        boolean isInUATMode = Utils.isTrue(this.objectFactory.<Boolean>get(Constants.MerchantMeta.IS_UAT));
        final String availabilityCheckUrl = NetworkConstants.getApiBaseUrl(isInUATMode) + NetworkConstants.APIEndPoints.API_AVAILABILITY_CHECK;
        final String body = prepareAvailabilityCheckAPIRequest(availabilityCheckRequest, sdkContext, phonePeContext).toJsonString();
        getCommonHeadersAdded(availabilityCheckRequest.getHeaderMap(), new OnHeadersAddedListener() {
            @Override
            public void onHeadersAdded(Map<String, String> addedHeaders) {
                APIHelper.this.apiManager.asyncPostRequest(availabilityCheckUrl, addedHeaders, body, listener, false);
            }
        });
    }

    public void syncSDKConfig(SDKContext sdkContext, PhonePeContext phonePeContext,@NonNull final INetworkResponseListener listener) {
        boolean isInUATMode = Utils.isTrue(objectFactory.<Boolean>get(Constants.MerchantMeta.IS_UAT));
        final String syncConfigEndPoint = NetworkConstants.getApiBaseUrl(isInUATMode) + NetworkConstants.APIEndPoints.API_SDK_CONFIG;
        Map<String, String> headers = objectFactory.<String, String>getHashMap();
        String sdkChecksum = Utils.getCheckSum(objectFactory,NetworkConstants.APIEndPoints.API_SDK_CONFIG,null);
        headers.put(Constants.GenericConstants.SDK_CHECKSUM,sdkChecksum);
        final String body = prepareAvailabilityCheckAPIRequest(sdkContext, phonePeContext).toJsonString();
        getCommonHeadersAdded(headers,new OnHeadersAddedListener() {
            @Override
            public void onHeadersAdded(Map<String, String> addedHeaders) {
                APIHelper.this.apiManager.asyncPostRequest(syncConfigEndPoint, addedHeaders, body, listener, false);
            }
        });
    }

    public void getTransactionRedirectUrl(TransactionRequest transactionRequest, SDKContext sdkContext, PhonePeContext phonePeContext, @NonNull final INetworkResponseListener listener) {
        String merchantProviderUrl = transactionRequest.getAPIUrl();
        boolean isInUATMode = Utils.isTrue(this.objectFactory.<Boolean>get(Constants.MerchantMeta.IS_UAT));
        String debitUrl = NetworkConstants.getApiBaseUrlWithVersion(isInUATMode) + "debit";
        if (merchantProviderUrl != null) {
            debitUrl = NetworkConstants.getApiBaseUrl(isInUATMode) + merchantProviderUrl;
        }
        final String finalDebitUrl = debitUrl;
        final String body = prepareRequest(transactionRequest, sdkContext, phonePeContext).toJsonString();


        getCommonHeadersAdded(transactionRequest.getHeaderMap(), new OnHeadersAddedListener() {
            @Override
            public void onHeadersAdded(Map<String, String> addedHeaders) {
                APIHelper.this.apiManager.asyncPostRequest(finalDebitUrl, addedHeaders, body, listener, false);
            }
        });
    }

    public void getProfileInfo(@NonNull final String apiUrl, @NonNull String checksum, @NonNull final INetworkResponseListener listener) {
        Map<String, String> headers = this.objectFactory.<String, String>getHashMap();
        headers.put(NetworkConstants.KEY_X_VERIFY, checksum);
        getCommonHeadersAdded(headers, new OnHeadersAddedListener() {
            @Override
            public void onHeadersAdded(Map<String, String> addedHeaders) {
                APIHelper.this.apiManager.asyncGetRequest(apiUrl, addedHeaders, null, listener, false);
            }
        });
    }

    public void logDeviceEvent(final DeviceEventPayload payload, String token) {
        Map<String, String> headers = this.objectFactory.getHashMap();
        headers.put(AnalyticsManager.EventConstants.X_AUTH_TOKEN, token);
        boolean isInUATMode = Utils.isTrue(this.objectFactory.<Boolean>get(Constants.MerchantMeta.IS_UAT));
        final String url = NetworkConstants.getDeviceLogUrl(isInUATMode);
        getCommonHeadersAdded(headers, new OnHeadersAddedListener() {
            @Override
            public void onHeadersAdded(Map<String, String> addedHeaders) {
                APIHelper.this.apiManager.asyncPostRequest(url, addedHeaders, payload.toJsonString(), null, false);
            }
        });
    }

    public boolean isNetworkAvailable() {
        ConnectivityManager cm =
                (ConnectivityManager) this.objectFactory.getApplicationContext().getSystemService(Context.CONNECTIVITY_SERVICE);
        NetworkInfo netInfo = cm.getActiveNetworkInfo();
        return netInfo != null && netInfo.isConnectedOrConnecting();
    }

    //*********************************************************************
    // Private methods
    //*********************************************************************

    private DebitInitRequest prepareRequest(TransactionRequest transactionRequest, SDKContext sdkContext, PhonePeContext phonePeContext) {

        DebitInitRequest debitInitRequest = this.objectFactory.<DebitInitRequest>get(DebitInitRequest.class);
        debitInitRequest.setData(transactionRequest.getData());
        debitInitRequest.setSdkContext(sdkContext);
        debitInitRequest.setPhonePeContext(phonePeContext);
        return debitInitRequest;
    }

    private AvailabilityCheckAPIRequest prepareAvailabilityCheckAPIRequest(AvailabilityCheckRequest availabilityCheckRequest, SDKContext sdkContext, PhonePeContext phonePeContext) {
        AvailabilityCheckAPIRequest availabilityCheckAPIRequest = this.objectFactory.get(AvailabilityCheckAPIRequest.class);
        availabilityCheckAPIRequest.setData(availabilityCheckRequest.getData());
        availabilityCheckAPIRequest.setSdkContext(sdkContext);
        availabilityCheckAPIRequest.setPhonePeContext(phonePeContext);
        return availabilityCheckAPIRequest;
    }

    private APIRequest prepareAvailabilityCheckAPIRequest(SDKContext sdkContext, PhonePeContext phonePeContext) {
        APIRequest sdkConfigAPIRequest = this.objectFactory.get(APIRequest.class);
        sdkConfigAPIRequest.setSdkContext(sdkContext);
        sdkConfigAPIRequest.setPhonePeContext(phonePeContext);
        return sdkConfigAPIRequest;
    }

    public void getCommonHeadersAdded(final Map<String, String> headers, final OnHeadersAddedListener listener) {

        headers.put(NetworkConstants.KEY_CONTENT_TYPE, NetworkConstants.KEY_CONTENT_TYPE_VALUE);
        headers.put(NetworkConstants.KEY_SOURCE_HEADER, NetworkConstants.KEY_SOURCE_HEADER_VALUE);
        headers.put(NetworkConstants.KEY_SOURCE_VERSION, BuildConfig.SDK_VERSION);
        headers.put(NetworkConstants.KEY_APP_VERSION, this.deviceInfoProvider.getAppVersion());
        headers.put(NetworkConstants.KEY_OS_VERSION, this.deviceInfoProvider.getOsVersion());
        headers.put(NetworkConstants.KEY_DEVICE_MODEL, this.deviceInfoProvider.getModel());
        headers.put(NetworkConstants.KEY_DEVICE_MANUFACTURER, this.deviceInfoProvider.getManufacturer());
        headers.put(NetworkConstants.KEY_MERCHANT_APP_ID, this.deviceInfoProvider.getPackageName());
        if(! Utils.isNullOrEmpty(this.deviceInfoProvider.getAppId()) ) {
            headers.put(NetworkConstants.KEY_APP_ID, this.deviceInfoProvider.getAppId());
        }
        headers.put(SESSION_ID, this.objectFactory.getSessionId());

        this.deviceInfoProvider.getDeviceFingerPrint(new iDeviceIdListener() {
            @Override
            public void onDeviceIdAvailable(String deviceId) {
                if(deviceId != null && !deviceId.isEmpty()) {
                    headers.put(NetworkConstants.KEY_DEVICE_ID, deviceId);
                    headers.put(NetworkConstants.KEY_DEVICE_UPI_ID, APIHelper.this.deviceInfoProvider.getUpiDeviceIdFromDeviceId(deviceId));
                }
                listener.onHeadersAdded(headers);
            }
        });
    }


    public interface OnHeadersAddedListener {
        void onHeadersAdded(Map<String, String> addedHeaders);
    }

    //*********************************************************************
    // End of class
    //*********************************************************************

}
