package com.phonepe.intent.sdk.mvp;

import android.content.Intent;
import android.net.Uri;
import android.net.http.HttpResponseCache;
import android.os.Bundle;
import android.os.Handler;
import android.support.annotation.NonNull;

import com.phonepe.intent.sdk.api.TransactionRequest;
import com.phonepe.intent.sdk.core.EventLoggerJS;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;
import com.phonepe.intent.sdk.models.Body;
import com.phonepe.intent.sdk.models.Event;
import com.phonepe.intent.sdk.models.ShowLoaderState;
import com.phonepe.intent.sdk.models.IntentResponse;
import com.phonepe.intent.sdk.models.JSLoadState;
import com.phonepe.intent.sdk.models.Response;
import com.phonepe.intent.sdk.models.SDKConfig;
import com.phonepe.intent.sdk.models.SDKContext;
import com.phonepe.intent.sdk.models.UrlConfig;
import com.phonepe.intent.sdk.networking.APIHelper;
import com.phonepe.intent.sdk.networking.INetworkResponseListener;
import com.phonepe.intent.sdk.networking.NetworkConstants;
import com.phonepe.intent.sdk.networking.models.PhonePeContext;
import com.phonepe.intent.sdk.networking.models.RedirectResponse;
import com.phonepe.intent.sdk.ui.TransactionActivity;
import com.phonepe.intent.sdk.utils.AnalyticsManager;
import com.phonepe.intent.sdk.utils.Config;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

import org.json.JSONObject;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.ACTION;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.BACK_PRESS_EVENT_INFO;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.INTENT_URL;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.RESULT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.SDK_TRANSACTION_STATUS;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.EventConstants.WAS_CANCELED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.BACK_PRESSED_EVENT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.PHONEPE_APP_OPENED_FOR_RESULT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.PHONEPE_APP_RETURNED_RESULT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_BACK_BUTTON_CLICKED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_BACK_CANCELLED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_BACK_CONFIRMED;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_MERCHANT_CALLBACK_SENT;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_PAGE_LOAD_COMPLETE;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_RENDER_START;
import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_TRANSACTION_TOKEN_RECEIVED;

/**
 * @author TheEternalWitness
 * @since 31/03/18.
 */
public class TransactionPresenter implements iTransactionPresenter, ObjectFactoryInitializationStrategy, INetworkResponseListener {

    public static final String TAG = "TransactionPresenter";
    private int retryLimit;
    private iTransactionView transactionView;
    private boolean isJSLoaded = false;
    private String urlToSave;
    private RedirectResponse redirectResponse;
    private TransactionRequest transactionRequest;
    private SDKContext sdkContext;
    private APIHelper apiHelper;
    private EventLoggerJS eventLoggerJS;
    private String payRedirectUrl;
    private int retryCount;
    private ObjectFactory objectFactory;
    private AnalyticsManager analyticsManager;
    private boolean isDeeplinkLaunched = false;

    protected ObjectFactory getObjectFactory() {

        return this.objectFactory;
    }

    protected EventLoggerJS getEventLoggerJS() {

        return this.eventLoggerJS;
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        this.apiHelper = objectFactory.<APIHelper>get(APIHelper.class);
        this.transactionView = initializationBundle.<iTransactionView>get(TransactionActivity.TRX_VIEW, null);
        this.eventLoggerJS = objectFactory.<EventLoggerJS>get(EventLoggerJS.class, initializationBundle);
        this.retryLimit = objectFactory.<Config>get(Config.class).getTransactionPresenterRetryLimit();
        this.objectFactory = objectFactory;
        this.analyticsManager = objectFactory.<AnalyticsManager>get(AnalyticsManager.class);
    }

    @Override
    public boolean isCachingAllowed() {
        return false;
    }

    //*********************************************************************
    // View related APIs
    //*********************************************************************

    @Override
    public void onRequestReceived(Intent startIntent, Bundle savedInstanceState) {

        this.transactionRequest = this.<TransactionRequest>getFromBundleOrIntent(savedInstanceState, startIntent, Constants.BundleConstants.KEY_REQUEST);
        this.redirectResponse = this.<RedirectResponse>getFromBundleOrIntent(savedInstanceState, startIntent, Constants.BundleConstants.KEY_DEBIT_RESPONSE);
        this.urlToSave = this.<String>getParcelableFromSavedBundle(savedInstanceState, Constants.BundleConstants.KEY_LAST_URL);
        this.sdkContext = this.<SDKContext>getFromBundleOrIntent(savedInstanceState, startIntent, Constants.BundleConstants.KEY_SDK_CONTEXT);
        Boolean deeplinkLaunched = this.<Boolean>getFromBundleOrIntent(savedInstanceState, startIntent, Constants.BundleConstants.KEY_DEEPLINK_LAUNCHED);
        if(deeplinkLaunched != null) {
            isDeeplinkLaunched = deeplinkLaunched;
        }
        this.transactionView.setLoadingStatus(true);

        if(isDeeplinkLaunched) {
            return;
        } else if (urlToSave != null && !urlToSave.isEmpty()) {
            this.transactionView.openUrlInWebView(urlToSave);

        } else if (this.redirectResponse != null) {

            handleRedirect(this.redirectResponse);

        } else if (this.transactionRequest != null) {

            handleRedirect(this.transactionRequest);
        } else {
            showRetry(Constants.ErrorCodes.ERROR_INVALID_DATA, true);
        }
    }

    private <T> T getFromBundleOrIntent(Bundle bundle, Intent intent, String key) {

        T object = this.<T>getParcelableFromSavedBundle(bundle, key);
        return object != null ? object : this.<T>getParcleableFromIntent(intent, key);
    }

    private <T> T getParcleableFromIntent(Intent intent, String key) {

        if (intent != null && key != null && intent.getExtras() != null) {

            return (T) intent.getParcelableExtra(key);
        }
        return null;
    }

    private <T> T getParcelableFromSavedBundle(Bundle bundle, String key) {

        if (bundle != null && key != null && bundle.containsKey(key)) {
            return (T) bundle.get(key);
        }

        return null;
    }

    public TransactionRequest getTransactionRequest() {
        return this.transactionRequest;
    }

    public RedirectResponse getRedirectResponse() {
        return this.redirectResponse;
    }

    public String getUrlToSave() {
        return this.urlToSave;
    }

    public int getRetryCount() {

        return this.retryCount;
    }

    @Override
    public void onBackPressed() {
        if (this.transactionView == null) {
            SdkLogger.w(TAG, String.format(Constants.LogFormats.FUNCTION_EARLY_EXIT_WARRNING, "transactionView", "onBackPressed"));
            return;
        }
        sendEvent(SDK_BACK_BUTTON_CLICKED);
        if (isJSLoaded) {
            sendBackPressedEvent();
        } else {
            this.transactionView.showAlert();
        }
    }

    @Override
    public void saveInstanceState(Bundle bundle) {

        String key = "bundle";
        if (Utils.isNull(bundle, TransactionPresenter.TAG, key)) {
            return;
        }

        bundle.putParcelable(Constants.BundleConstants.KEY_REQUEST, this.transactionRequest);
        bundle.putParcelable(Constants.BundleConstants.KEY_DEBIT_RESPONSE, this.redirectResponse);
        bundle.putString(Constants.BundleConstants.KEY_LAST_URL, urlToSave);
        bundle.putParcelable(Constants.BundleConstants.KEY_SDK_CONTEXT, this.sdkContext);
        bundle.putBoolean(Constants.BundleConstants.KEY_DEEPLINK_LAUNCHED, this.isDeeplinkLaunched);

    }

    @Override
    public void onDestroy() {
        this.transactionView = null;
    }


    public iTransactionView getTransactionView() {
        return transactionView;
    }

    @Override
    public void onBackPressConfirmed() {
        if (this.transactionView != null) {
            sendEvent(SDK_BACK_CONFIRMED);
            announceCancel();
        }
    }

    @Override
    public void onBackPressCancelled() {
        if (this.transactionView != null) {
            sendEvent(SDK_BACK_CANCELLED);
        }
    }

    @Override
    public void onRetryPressed() {

        if (Utils.isNull(this.transactionView, TransactionPresenter.TAG, "transactionView")) {

            return;
        }
        this.transactionView.setLoadingStatus(true);
        retryCount++;
        getUrlFromServerAndRedirect(this.transactionRequest);

    }

    @Override
    public void onActivityResult(boolean wasCanceled, IntentResponse intentResponse) {
        sendIntentResultReceivedEvent(wasCanceled, intentResponse);
    }

    //*********************************************************************
    // Bridge related
    //*********************************************************************

    @Override
    public void onJSLoadStateChanged(String context, String isJSLoaded, String callback) {
        JSLoadState jsLoadState = JSLoadState.fromJsonString(isJSLoaded, this.objectFactory, JSLoadState.class);
        this.isJSLoaded = jsLoadState.isJSLoaded();
        if (this.transactionView != null) {
            Body body = this.objectFactory.<Body>get(Body.class);
            this.transactionView.onBridgeCallBack(callback, null, this.objectFactory.getResponse(Constants.GenericConstants.SUCCESS).toJsonString(), context, body.toJsonString());
        }
    }

    @Override
    public void onShowLoader(String context, String showLoader, String callback) {
        ShowLoaderState showLoaderState = ShowLoaderState.fromJsonString(showLoader, this.objectFactory, ShowLoaderState.class);
        boolean shouldShowLoader = showLoaderState.showLoader();
        if(this.transactionView != null) {
            this.transactionView.setLoadingStatus(shouldShowLoader);
            Body body = this.objectFactory.<Body>get(Body.class);
            this.transactionView.onBridgeCallBack(callback, null, this.objectFactory.getResponse(Constants.GenericConstants.SUCCESS).toJsonString(), context, body.toJsonString());
        }
    }

    @Override
    public void onComplete(String result) {
        if (Utils.isNull(this.transactionView, TransactionPresenter.TAG, "transactionView")) {

            return;
        }

        sendEvent(SDK_MERCHANT_CALLBACK_SENT);
        Response response = Response.fromJsonString(result, this.objectFactory, Response.class);
        if (response != null && response.<String>get(Response.KEY_STATUS_CODE) != null && Constants.GenericConstants.USER_CANCEL.matches(response.<String>get(Response.KEY_STATUS_CODE))) {
            this.transactionView.endWithCancel(this.objectFactory.getResponse(Constants.GenericConstants.USER_CANCEL).toJsonString());
        } else {
            this.transactionView.endWithResult(result);
        }

    }

    private void sendEvent(String eventName) {

        Event event = this.objectFactory.<AnalyticsManager>get(AnalyticsManager.class).getEvent(eventName);
        sendEvent(event);
    }

    private void sendEvent(Event event) {

        this.objectFactory.<AnalyticsManager>get(AnalyticsManager.class).submit(event);
    }

    @Override
    public void keepUrl(UrlConfig urlConfig) {
        if (urlConfig != null && this.transactionView != null) {
            boolean isInUatMode = Utils.isTrue(this.objectFactory.<Boolean>get(Constants.MerchantMeta.IS_UAT));
            urlToSave = urlConfig.getFinalUrl(NetworkConstants.getApiBaseUrl(isInUatMode));
        }
    }

    @Override
    public void onJusPayTransactionCompleted(String transactionId, String status, boolean isSuccess) {
        SdkLogger.v(TAG, "JustPay Completed");
        if (isSuccess) {
            onComplete(status);
        } else {
            this.transactionView.endWithCancel(status);
        }
    }

    //*********************************************************************
    // URL monitoring
    //*********************************************************************

    @Override
    public void onPageLoadStart(String url) {
        if (this.transactionView != null && url.equals(payRedirectUrl)) {

            this.analyticsManager.submit(this.analyticsManager.getEvent(SDK_RENDER_START));
        }
    }

    @Override
    public void onPageFinished(String presentUrl) {
        // Keep domain start with "."
        if (presentUrl.toLowerCase().startsWith(NetworkConstants.HTTPS) && presentUrl.toLowerCase().contains(NetworkConstants.PHONEPE_COM) &&
                !presentUrl.toLowerCase().contains(NetworkConstants.JAVASCRIPT)) {
            urlToSave = presentUrl;
        }
        if (this.transactionView != null && presentUrl.equals(payRedirectUrl)) {
            sendEvent(SDK_PAGE_LOAD_COMPLETE);
        }
    }

    //*********************************************************************
    // API related
    //*********************************************************************

    private void getUrlFromServerAndRedirect(TransactionRequest transactionRequest) {
        SdkLogger.d("V3/DEBIT", "Starting v3/debit call");
        SDKContext sdkContext = getSDKContext();
        sdkContext.setPhonePeAppData();
        this.apiHelper.getTransactionRedirectUrl(transactionRequest, sdkContext, getPhonePeContext(), this);
    }

    //*********************************************************************
    // Utility methods
    //*********************************************************************

    private void handleRedirect(TransactionRequest transactionRequest) {

        if (transactionRequest.getRedirectUrl() != null) {
            this.transactionView.openUrlInWebView(transactionRequest.getRedirectUrl());
        } else {
            getUrlFromServerAndRedirect(transactionRequest);
        }
    }

    private RedirectResponse getRedirectResponse(String res) {

        if (Utils.isNull(res, TransactionPresenter.TAG, "res")) {
            return null;
        }

        return RedirectResponse.fromJsonString(res, this.objectFactory, RedirectResponse.class);
    }

    private void readSDKConfig(String response) {

        try {
            SDKConfig sdkConfig = getObjectFactory().get(SDKConfig.class);
            JSONObject jsonObject = new JSONObject(response);
            try {
                if(jsonObject.has("data")) {
                    sdkConfig.readConfig(jsonObject.getJSONObject("data"));
                }
            } catch (Throwable t) {
                SdkLogger.e(TAG, "Error Reading SDK Config");
            }
            if(sdkConfig.isCacheReportingEnabled()) {
                Event event = analyticsManager.getEvent(AnalyticsManager.Events.SDK_CACHE_METRICS);
                if(Utils.isTrue(getObjectFactory().<Boolean>get(Constants.MerchantMeta.PRE_CACHE_ENABLED))) {
                    HttpResponseCache cache = HttpResponseCache.getInstalled();
                    event.putInEventBody("requestCount", cache.getRequestCount())
                            .putInEventBody("hitCount", cache.getHitCount())
                            .putInEventBody("networkCount", cache.getNetworkCount())
                            .putInEventBody("size", cache.size())
                            .putInEventBody("maxSize", cache.maxSize())
                            .putInEventBody("preCacheEnabled", true);
                } else {
                    event.putInEventBody("preCacheEnabled", false);
                }
                analyticsManager.submit(event);
            }
        } catch (Exception e) {
            SdkLogger.e(TAG, e.getMessage(), e);
        }
    }

    private void handleRedirect(RedirectResponse redirectResponse) {
        if (redirectResponse.isSuccess()) {
            registerToken();
            switch (redirectResponse.getRedirectType()) {
                case WEB:
                    this.transactionView.openUrlInWebView(redirectResponse.getRedirectUrl());
                    break;
                case INTENT:
                    String intentUrl = redirectResponse.getRedirectUrl();
                    if (!this.isDeeplinkLaunched) {
                        SdkLogger.v(TAG, "Opening PP App with Url: " + intentUrl);
                        sentIntentRedirectEvent(intentUrl);
                        if (Utils.isUPIUrl(intentUrl)) {
                            this.isDeeplinkLaunched = true;
                            this.transactionView.openActivityForResult(Uri.parse(intentUrl));
                        } else {
                            showRetry("Invalid redirection information.", true);
                            this.transactionView.setLoadingStatus(false);
                        }
                    }

            }
        } else {
            String message = "Something went wrong";
            try {
                message = redirectResponse.getMessage();
            } catch (Exception e) {

            }
            retryLimitCrossed(message);
        }
    }

    private void registerToken() {
        sendEvent(SDK_TRANSACTION_TOKEN_RECEIVED);
    }

    private void sendBackPressedEvent() {
        Event event = this.objectFactory.<AnalyticsManager>get(AnalyticsManager.class).getEvent(BACK_PRESSED_EVENT);
        event.putInEventBody(ACTION, BACK_PRESS_EVENT_INFO);
        this.eventLoggerJS.sendEventToJS(event.toJsonString());
    }

    private void announceCancel() {

        Response response = this.objectFactory.getResponse(Constants.ErrorCodes.ERROR_CANCELED);
        this.transactionView.endWithCancel(response.toJsonString());

        AnalyticsManager analyticsManager = this.objectFactory.<AnalyticsManager>get(AnalyticsManager.class);
        Event event = analyticsManager.getEvent(SDK_MERCHANT_CALLBACK_SENT)
                .putInEventBody(SDK_TRANSACTION_STATUS, Constants.ErrorCodes.ERROR_CANCELED);

        sendEvent(event);

    }

    private void showRetry(String errorMessage, boolean showButton) {
        if (this.transactionView != null) {
            this.transactionView.showError(errorMessage, showButton);
        }
    }

    private void retryLimitCrossed(String errorMessage) {
        this.transactionView.showError(errorMessage, false);
        Handler handler = this.objectFactory.getAndroidHandler();
        handler.postDelayed(new Runnable() {
            @Override
            public void run() {
                onComplete(Constants.GenericConstants.FAILURE);
            }
        }, 1500);
    }

    //*********************************************************************
    // Event related
    //*********************************************************************

    private void sentIntentRedirectEvent(String intentUrl) {

        AnalyticsManager analyticsManager = this.objectFactory.<AnalyticsManager>get(AnalyticsManager.class);
        Event event = analyticsManager.getEvent(PHONEPE_APP_OPENED_FOR_RESULT)
                .putInEventBody(INTENT_URL, intentUrl);
        sendEvent(event);
    }

    private void sendIntentResultReceivedEvent(boolean wasCanceled, IntentResponse intentResponse) {

        AnalyticsManager analyticsManager = this.objectFactory.<AnalyticsManager>get(AnalyticsManager.class);
        Event event = analyticsManager.getEvent(PHONEPE_APP_RETURNED_RESULT)
                .putInEventBody(WAS_CANCELED, String.valueOf(wasCanceled));
        if (intentResponse != null) {
            event.putInEventBody(RESULT, intentResponse.toString());
        }

        sendEvent(event);
    }

    protected SDKContext getSDKContext() {

        return this.sdkContext;
    }

    protected PhonePeContext getPhonePeContext() {
        return null;
    }

    @Override
    public void onSuccess(String response) {
        SdkLogger.d("V3/DEBIT", "Got v3/debit response");
        if (Utils.isNull(this.transactionView, TransactionPresenter.TAG, "transactionView")) {

            return;
        }
        readSDKConfig(response);
        this.redirectResponse = getRedirectResponse(response);
        if (this.redirectResponse == null) {
            handleRetryAndError(Constants.ErrorCodes.ERROR_PARSE_ERROR, true);
            this.transactionView.setLoadingStatus(false);
            return;
        }

        triggerWorkflow(redirectResponse);
    }

    private void triggerWorkflow(@NonNull RedirectResponse redirectResponse) {

        handleRedirect(redirectResponse);

//        switch (redirectResponse.getSdkFlow()) {
//
//            case INTENT:
//            case EMBEDDED:
//                handleRedirect(redirectResponse);
//                break;
//        }
    }

    private void handleRetryAndError(String errorMessage, boolean showButton) {

        if (retryCount >= TransactionPresenter.this.retryLimit) {
            retryLimitCrossed(Constants.ErrorCodes.ERROR_PARSE_ERROR);
        } else {
            showRetry(errorMessage, showButton);
        }
    }

    @Override
    public void onFailure(int responseCode, String response) {

        if (Utils.isNull(this.transactionView, TransactionPresenter.TAG, "transactionView")) {

            return;
        }
        boolean isNetworkAvailable = this.apiHelper.isNetworkAvailable();
        String errorMessage = isNetworkAvailable ? Constants.ErrorCodes.ERROR_PARSE_ERROR : "Network unavailable.";
        handleRetryAndError(errorMessage, true);
        //this.transactionView.setLoadingStatus(false);
    }


    //*********************************************************************
    // End of class
    //*********************************************************************
}
