package com.phonepe.intent.sdk.models;


import com.phonepe.intent.sdk.core.DataConfig;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;
import com.phonepe.intent.sdk.utils.GenericJSONUtils;
import com.phonepe.intent.sdk.utils.SdkLogger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

public class SDKConfig implements ObjectFactoryInitializationStrategy {


    private static String TAG = "SDKConfig";

    private static String KEY_PRECACHE_ENABLED = "isPrecacheEnabled";
    private static String KEY_PRECACHE_URL = "precacheUrl";
    private static String KEY_USE_PRECACHE = "usePrecache";
    private static String KEY_WEBVIEW_CACHE_ENABLED = "isWebViewCacheEnabled";
    private static String KEY_APPROVED_FILE_TYPES = "fileTypes";
    private static String KEY_CACHE_REPORTING_ENABLED = "isCacheReportingEnabled";
    private static String KEY_WEB_LOGS_ENABLED = "areWebLogsEnabled";
    private static String KEY_WEB_LOGS_LEVEL = "webLogsLevel";
    private static String KEY_WEB_LOGS_LOCATION = "webLogsReportingLocation";
    private static String KEY_ANDROID_LOGS_ENABLED = "areAndroidLogsEnabled";
    private static String KEY_ANDROID_LOGS_LEVEL = "androidLogsLevel";
    private static String KEY_ANDROID_LOGS_LOCATION = "androidLogsReportingLocation";
    private static String KEY_SDK_TO_APP_COMMUNICATION = "sdkToAppCommunication";
    private static String KEY_SHOW_PHONEPE_ENABLED = "showPhonePeEnabled";

    private static String KEY_SDK_CONFIG = "sdkConfig";
    private static String KEY_PRECACHE = "precache";
    private static String KEY_ENABLED = "enabled";
    private static String KEY_URL = "url";
    private static String KEY_CACHE = "cache";
    private static String KEY_WEB_CACHE_ENABLED = "webviewCacheEnabled";
    private static String KEY_ANDROID_LOGS = "androidLogs";
    private static String KEY_LOG_LEVEL = "logLevel";
    private static String KEY_REPORTING_LEVEL = "reportingLevel";
    private static String KEY_CACHE_METRICS_REPORTING = "cacheMetricsReporting";
    private static String KEY_WEB_LOGS = "webLogs";

    public static int LOG_LEVEL_DEBUG = 0;
    public static int LOG_LEVEL_INFO = 1;
    public static int LOG_LEVEL_WARN = 2;
    public static int LOG_LEVEL_ERROR = 3;

    public static int REPORTING_LEVEL_REMOTE = 1;
    public static int REPORTING_LEVEL_LOCAL = 0;

    private boolean precacheEnabled;
    private String precacheUrl;
    private boolean usePreCache;
    private boolean enableWebViewCache;
    private List<String> approvedFileTypes;
    private boolean cacheReportingEnabled;
    private boolean webLogsEnabled;
    private int webLogsLevel;
    private boolean androidLogsEnabled;
    private int androidLogsLevel;
    private DataConfig dataConfig;
    private ObjectFactory objectFactory;


    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {
        this.objectFactory = objectFactory;
        dataConfig = objectFactory.<DataConfig>get(DataConfig.class);
    }

    public void readConfig(JSONObject data) {
        if (data != null) {
            try {
                if (data.has(KEY_SDK_CONFIG)) {
                    JSONObject config = data.getJSONObject(KEY_SDK_CONFIG);
                    JSONObject precache = GenericJSONUtils.readJSONObject(config, KEY_PRECACHE, false, false);
                    if (precache != null) {
                        if (precache.has(KEY_ENABLED)) {
                            setPrecacheEnabled(GenericJSONUtils.readBoolean(precache, KEY_ENABLED, false, false));
                        }
                        setPrecacheUrl(GenericJSONUtils.readString(precache, KEY_URL, false, false));
                    }
                    if (config.has(KEY_CACHE)) {
                        JSONObject cache = config.getJSONObject(KEY_CACHE);
                        if (cache.has(KEY_USE_PRECACHE))
                            setUsePreCache(cache.getBoolean(KEY_USE_PRECACHE));

                        if (cache.has(KEY_WEB_CACHE_ENABLED))
                            setEnableWebViewCache(cache.getBoolean(KEY_WEB_CACHE_ENABLED));

                        if (cache.has(KEY_APPROVED_FILE_TYPES))
                            setApprovedFileTypes(cache.getJSONArray(KEY_APPROVED_FILE_TYPES).toString());
                    }
                    if (config.has(KEY_ANDROID_LOGS)) {
                        JSONObject androidLogs = config.getJSONObject(KEY_ANDROID_LOGS);
                        if (androidLogs.has(KEY_ENABLED))
                            setAndroidLogsEnabled(androidLogs.getBoolean(KEY_ENABLED));
                        if (androidLogs.has(KEY_LOG_LEVEL)) {
                            String logLevel = androidLogs.getString(KEY_LOG_LEVEL);
                            setAndroidLogsLevel(getLogLevel(logLevel));
                        }
                        if (androidLogs.has(KEY_REPORTING_LEVEL)) {
                            setAndroidLogsReportLocation(getReportingLevel(androidLogs));
                        }
                    }

                    if (config.has(KEY_WEB_LOGS)) {
                        JSONObject webLogs = config.getJSONObject(KEY_WEB_LOGS);
                        if (webLogs.has(KEY_ENABLED)) {
                            setWebLogsEnabled(webLogs.getBoolean(KEY_ENABLED));
                        }
                        if (webLogs.has(KEY_LOG_LEVEL)) {
                            String logLevel = webLogs.getString(KEY_LOG_LEVEL);
                            setWebLogsLevel(getLogLevel(logLevel));
                        }
                        if (webLogs.has(KEY_REPORTING_LEVEL)) {
                            setWebLogsReportLocation(getReportingLevel(webLogs));
                        }
                    }

                    if (config.has(KEY_CACHE_METRICS_REPORTING)) {
                        JSONObject cacheReporting = config.getJSONObject(KEY_CACHE_METRICS_REPORTING);
                        if (cacheReporting.has(KEY_ENABLED)) {
                            setCacheReportingEnabled(cacheReporting.getBoolean(KEY_ENABLED));
                        }
                    }

                    if(config.has(KEY_SDK_TO_APP_COMMUNICATION)) {
                        JSONObject sdkToAppCommunication = config.getJSONObject(KEY_SDK_TO_APP_COMMUNICATION);
                        if(sdkToAppCommunication.has(KEY_SHOW_PHONEPE_ENABLED)) {
                            setShowPhonepeEnabled(sdkToAppCommunication.getBoolean(KEY_SHOW_PHONEPE_ENABLED));
                        }
                    }

                }
            } catch (Exception e) {
                SdkLogger.e(TAG, e.getMessage(), e);
            }
        }
    }

    private int getLogLevel(String logLevel) {
        int logLevelInt = LOG_LEVEL_ERROR;
        if (logLevel.matches("DEBUG")) {
            logLevelInt = LOG_LEVEL_DEBUG;
        } else if (logLevel.matches("ERROR")) {
            logLevelInt = LOG_LEVEL_ERROR;
        } else if (logLevel.matches("INFO")) {
            logLevelInt = LOG_LEVEL_INFO;
        } else if (logLevel.matches("WARN")) {
            logLevelInt = LOG_LEVEL_WARN;
        }
        return logLevelInt;
    }

    private int getReportingLevel(JSONObject jsonObject) throws JSONException {
        int reportingLocation = REPORTING_LEVEL_LOCAL;
        if (jsonObject.getString(KEY_REPORTING_LEVEL).matches("REMOTE")) {
            reportingLocation = REPORTING_LEVEL_REMOTE;
        }
        return reportingLocation;
    }

    @Override
    public boolean isCachingAllowed() {
        return true;
    }

    public boolean isPrecacheEnabled() {
        return dataConfig.getBool(KEY_PRECACHE_ENABLED, true);
    }

    public void setPrecacheEnabled(boolean precacheEnabled) {
        dataConfig.saveBool(KEY_PRECACHE_ENABLED, precacheEnabled);
    }

    public String getPrecacheUrl() {
        return dataConfig.getString(KEY_PRECACHE_URL, null);
    }

    public void setPrecacheUrl(String precacheUrl) {
        dataConfig.saveString(KEY_PRECACHE_URL, precacheUrl);
    }

    public boolean shouldUsePreCache() {
        return dataConfig.getBool(KEY_USE_PRECACHE, true);
    }

    public void setUsePreCache(boolean usePreCache) {
        dataConfig.saveBool(KEY_USE_PRECACHE, usePreCache);
    }

    public boolean isEnableWebViewCache() {
        return dataConfig.getBool(KEY_WEBVIEW_CACHE_ENABLED, true);
    }

    public void setEnableWebViewCache(boolean enableWebViewCache) {
        dataConfig.saveBool(KEY_WEBVIEW_CACHE_ENABLED, enableWebViewCache);
    }

    public List<String> getApprovedFileTypes() {
        approvedFileTypes = new ArrayList<>();
        String data = dataConfig.getString(KEY_APPROVED_FILE_TYPES, "");
        if (data != null && !data.isEmpty()) {
            try {
                JSONArray dataArray = new JSONArray(data);
                for (int i = 0; i < dataArray.length(); i++) {
                    approvedFileTypes.add(dataArray.getString(i));
                }
            } catch (JSONException e) {
                SdkLogger.e(TAG, e.getMessage(), e);
            }
        }
        return approvedFileTypes;
    }

    public void setApprovedFileTypes(String approvedFileTypes) {
        dataConfig.saveString(KEY_APPROVED_FILE_TYPES, approvedFileTypes);
    }

    public boolean isCacheReportingEnabled() {
        return dataConfig.getBool(KEY_CACHE_REPORTING_ENABLED, false);
    }

    public void setCacheReportingEnabled(boolean cacheReportingEnabled) {
        dataConfig.saveBool(KEY_CACHE_REPORTING_ENABLED, cacheReportingEnabled);
    }

    public boolean isWebLogsEnabled() {
        return dataConfig.getBool(KEY_WEB_LOGS_ENABLED, false);
    }

    public void setWebLogsEnabled(boolean webLogsEnabled) {
        dataConfig.saveBool(KEY_WEB_LOGS_ENABLED, webLogsEnabled);
    }

    public int getWebLogsLevel() {
        return dataConfig.getInt(KEY_WEB_LOGS_LEVEL, LOG_LEVEL_ERROR);
    }

    public void setWebLogsLevel(int webLogsLevel) {
        dataConfig.saveInteger(KEY_WEB_LOGS_LEVEL, webLogsLevel);
    }

    public void setWebLogsReportLocation(int webLogsReportingLocation) {
        dataConfig.saveInteger(KEY_WEB_LOGS_LOCATION, webLogsReportingLocation);
    }

    public int getWebLogsReportLocation() {
        return dataConfig.getInt(KEY_WEB_LOGS_LOCATION, REPORTING_LEVEL_LOCAL);
    }

    public boolean isAndroidLogsEnabled() {
        return dataConfig.getBool(KEY_ANDROID_LOGS_ENABLED, false);
    }

    public void setAndroidLogsEnabled(boolean androidLogsEnabled) {
        dataConfig.saveBool(KEY_ANDROID_LOGS_ENABLED, androidLogsEnabled);
    }

    public int getAndroidLogsLevel() {
        return dataConfig.getInt(KEY_ANDROID_LOGS_LEVEL, LOG_LEVEL_ERROR);
    }

    public void setAndroidLogsLevel(int androidLogsLevel) {
        dataConfig.saveInteger(KEY_ANDROID_LOGS_LEVEL, androidLogsLevel);
    }

    public void setAndroidLogsReportLocation(int androidLogsReportingLocation) {
        dataConfig.saveInteger(KEY_ANDROID_LOGS_LOCATION, androidLogsReportingLocation);
    }

    public int getAndroidLogsReportLocation() {
        return dataConfig.getInt(KEY_ANDROID_LOGS_LOCATION, REPORTING_LEVEL_LOCAL);
    }

    public void setShowPhonepeEnabled(boolean enabled) {
        dataConfig.saveBool(KEY_SHOW_PHONEPE_ENABLED, enabled);
    }

    public boolean shouldShowPhonePeEnabled() {
        return dataConfig.getBool(KEY_SHOW_PHONEPE_ENABLED, true);
    }
}
