package com.phonepe.intent.sdk.models;

import android.support.annotation.NonNull;

import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.RuntimeExceptionManager;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public abstract class AbstractJson implements ObjectFactoryInitializationStrategy {

    public static final String TAG = "AbstractJson";


    protected abstract JSONObject getJsonObject();
    protected abstract ObjectFactory getObjectFactory();

    public static <T> T get(JSONObject jsonObject, String key) {

        try {
            return (T) jsonObject.get(key);
        } catch (JSONException e) {

            SdkLogger.e(AbstractJson.TAG, String.format(Constants.LogFormats.JSON_EXCEPTION, e.getMessage(), key));
        }

        return null;
    }

    public static <T> T get(JSONArray jsonArray, int index) {

        try {
            return (T) jsonArray.get(index);
        } catch (JSONException e) {
            SdkLogger.e(AbstractJson.TAG, String.format(Constants.LogFormats.JSON_EXCEPTION, e.getMessage(), index));
        }
        return null;
    }

    public <T> T get(String key) {
        return get(getJsonObject(), key);
    }

    public boolean has(@NonNull String key){
        return getJsonObject().has(key);
    }

    public String toJsonString() {

        if (Utils.isNull(getJsonObject(), AbstractJson.TAG, "jsonObject")) {

            getObjectFactory().getRuntimeExceptionManager().submit(AbstractJson.TAG, "json object should not be null", RuntimeExceptionManager.Severity.LOW);
        }

        return getJsonObject().toString();
    }

    public <T> void put(String key, T value) {

        put(getJsonObject(), getObjectFactory(), key, value);
    }

    public static <T> void put(JSONObject jsonObject, ObjectFactory objectFactory, String key, T value) {

        if (Utils.isNull(jsonObject, AbstractJson.TAG, "jsonObject")) {

            objectFactory.getRuntimeExceptionManager().submit(AbstractJson.TAG, "json object should not be null", RuntimeExceptionManager.Severity.LOW);
        }

        try {

            jsonObject.<T>put(key, value);
        } catch (JSONException e) {

            SdkLogger.e(AbstractJson.TAG, String.format(Constants.LogFormats.JSON_EXCEPTION, e.getMessage(), key));
        }
    }

    public static <T extends ObjectFactoryInitializationStrategy> T fromJsonString(String jsonString, ObjectFactory objectFactory, Class<T> tClass) {

        if (objectFactory == null) {

            SdkLogger.e(AbstractJson.TAG, String.format(Constants.LogFormats.OBJECT_IS_NULL_OR_EMPTY, "objectFactory"));
            return null;
        }

        if (jsonString == null || jsonString.length() == 0) {

            SdkLogger.e(AbstractJson.TAG, String.format(Constants.LogFormats.OBJECT_IS_NULL_OR_EMPTY, "jsonString"));
            return null;
        }

        try {

            ObjectFactory.InitializationBundle initializationBundle = objectFactory.<ObjectFactory.InitializationBundle>get(ObjectFactory.InitializationBundle.class);
            initializationBundle.put(TAG, new JSONObject(jsonString));
            return objectFactory.<T>get(tClass, initializationBundle);
        } catch (JSONException e) {

            SdkLogger.e(AbstractJson.TAG, String.format(Constants.LogFormats.JSON_EXCEPTION, e.getMessage(), jsonString));
        }

        return objectFactory.get(tClass);
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {
    }

    @Override
    public boolean isCachingAllowed() {
        return false;
    }
}
