package com.phonepe.intent.sdk.core;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.os.Bundle;
import android.os.Handler;
import android.os.Parcel;
import android.os.Parcelable;
import android.util.Base64;

import com.phonepe.intent.sdk.models.Response;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.RuntimeExceptionManager;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;

/**
 * @author TheEternalWitness
 * @since 10/04/18.
 */
public class ObjectFactory implements Parcelable {


    public static final String TAG = "ObjectFactory";
    private static final String TCLASS = "tClass";
    private static final String SIGNATURE_DIGEST = "signature_digest";

    // static to share between various instances
    public static Context applicationContext;
    private static String sessionId = UUID.randomUUID().toString();
    private static DataObjectFactoryCache dataObjectFactoryCache = new DataObjectFactoryCache();

    protected ObjectFactory(Parcel in) {
    }

    public static final Creator<ObjectFactory> CREATOR = new Creator<ObjectFactory>() {
        @Override
        public ObjectFactory createFromParcel(Parcel in) {
            return new ObjectFactory(in);
        }

        @Override
        public ObjectFactory[] newArray(int size) {
            return new ObjectFactory[size];
        }
    };

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
    }

    private static class DataObjectFactoryCache {

        private Map<String, Object> cache = new HashMap<>();

        private <T> String getKey(Class<T> tClass) {

            return tClass.getCanonicalName();
        }

        public <T> boolean contains(Class<T> tClass) {

            if (Utils.isNull(tClass, TAG, TCLASS)) {

                return false;
            }
            return this.cache.containsKey(getKey(tClass));
        }

        public boolean contains(String key) {

            if (Utils.isNull(key, TAG, "key")) {
                return false;
            }

            return this.cache.containsKey(key);
        }

        public <T> T get(Class<T> tClass) {

            return (T) this.cache.get(getKey(tClass));
        }

        public <T> T get(String key) {

            return (T) this.cache.get(key);
        }

        public <T> T add(Class<T> tClass, Object object) {

            if (!this.contains(tClass)) {

                synchronized (DataObjectFactoryCache.class) {

                    if (!this.contains(tClass)) {

                        this.cache.put(getKey(tClass), object);
                    }
                }
            }

            return get(tClass);
        }

        public <T> void add(String key, T value) {

            this.cache.put(key, value);
        }

        public void flush() {

            this.cache.clear();
        }
    }


    public ObjectFactory(Context context) {

        SdkLogger.v(TAG, "validating context provided to sdk ...");
        if (Utils.isNull(context, TAG, Constants.LogFormats.CONTEXT)
                || Utils.isNull(context.getApplicationContext(), TAG, Constants.LogFormats.APPLICATION_CONTEXT)) {

            throw new RuntimeException("Application context is required for initialization.");
        }

        SdkLogger.i(TAG, "context provided is valid");
        ObjectFactory.applicationContext = context.getApplicationContext();
    }

    public <T> boolean cache(String key, T value) {
        if (Utils.isNull(value, TAG, key)) {

            return false;
        }

        ObjectFactory.dataObjectFactoryCache.add(key, value);
        return true;
    }

    public <T> T get(String key) {

        if (ObjectFactory.dataObjectFactoryCache.contains(key)) {

            return ObjectFactory.dataObjectFactoryCache.get(key);
        }

        return null;
    }

    public <T extends ObjectFactoryInitializationStrategy> T get(Class<T> tClass) {

        if (ObjectFactory.dataObjectFactoryCache.contains(tClass)) {

            return ObjectFactory.dataObjectFactoryCache.get(tClass);
        }

        return cache(tClass, create(tClass, null));

    }

    public <T extends ObjectFactoryInitializationStrategy> T get(Class<T> tClass, InitializationBundle initializationBundle) {

        return create(tClass, initializationBundle);
    }

    private final <T extends ObjectFactoryInitializationStrategy> T cache(Class<T> tClass, T object) {

        return object.isCachingAllowed() ? ObjectFactory.dataObjectFactoryCache.add(tClass, object) : object;
    }

    private final <T extends ObjectFactoryInitializationStrategy> T create(Class<T> tClass, InitializationBundle initializationBundle) {

        try {

            T object = tClass.newInstance();
            object.init(this, initializationBundle);
            return object;

        } catch (IllegalAccessException e) {
            SdkLogger.e(TAG, String.format("IllegalAccessException for class = {%s} caught,exception message = {%s}. Make sure class has public default constructor available.", tClass.getName(), e.getMessage()), e);

        } catch (InstantiationException e) {
            SdkLogger.e(TAG, String.format("InstantiationException for class = {%s} caught,exception message = {%s}.", tClass.getName(), e.getMessage()), e);
        }

        getRuntimeExceptionManager().submit(TAG, Constants.LogFormats.RUNTIME_EXCEPTION, RuntimeExceptionManager.Severity.LOW);
        return null;
    }

    public Context getApplicationContext() {
        return ObjectFactory.applicationContext;
    }

    public void flushCache() {

        ObjectFactory.dataObjectFactoryCache.flush();
    }

    public RuntimeExceptionManager getRuntimeExceptionManager() {

        return this.<RuntimeExceptionManager>get(RuntimeExceptionManager.class);
    }

    public Handler getAndroidHandler() {

        return new Handler();
    }

    public JSONObject getJsonObject() {

        return new JSONObject();
    }

    public JSONObject getJsonObject(String jsonString) {

        try {
            return new JSONObject(jsonString);
        } catch (JSONException e) {

            SdkLogger.e(TAG, String.format(Constants.LogFormats.JSON_EXCEPTION, e.getMessage(), jsonString), e);
        }

        return null;
    }

    public String getSessionId() {
        return ObjectFactory.sessionId;
    }

    public JSONArray getJsonArray() {

        return new JSONArray();
    }

    public JSONArray getJsonArray(String jsonString) {

        try {
            return new JSONArray(jsonString);
        } catch (JSONException e) {
            SdkLogger.e(TAG, String.format(Constants.LogFormats.JSON_EXCEPTION, e.getMessage(), jsonString), e);
        }

        return null;
    }

    public <T> ArrayList<T> getArrayList() {

        return new ArrayList<>();
    }

    public <T> ConcurrentLinkedQueue<T> getConcurrentLinkedQueue() {

        return new ConcurrentLinkedQueue<>();
    }

    public String getUniqueId() {

        return UUID.randomUUID().toString();
    }

    public Bundle getInitializedBundle() {

        Bundle bundle = new Bundle();
        bundle.putParcelable(Constants.BundleConstants.DATA_FACTORY, this);
        return bundle;
    }

    public <T> Intent createIntent(Context context, Class<T> tClass, Bundle bundle) {

        Intent intent = new Intent(context, tClass);
        if (bundle != null) {
            intent.putExtras(bundle);
        }
        return intent;
    }

    public <T, U> HashMap<T, U> getHashMap() {

        return new HashMap<T, U>();
    }

    public Response getResponse(String statusCode) {

        Response response = get(Response.class);
        switch (statusCode) {

            case Constants.GenericConstants.SUCCESS:
                response.<String>put(Response.KEY_STATUS_CODE, Constants.GenericConstants.SUCCESS);
                break;
            case Constants.GenericConstants.FAILURE:
                response.<String>put(Response.KEY_STATUS_CODE, Constants.GenericConstants.FAILURE);
                break;
            case Constants.GenericConstants.EXPIRED:
                response.<String>put(Response.KEY_STATUS_CODE, Constants.GenericConstants.EXPIRED);
                break;
            case Constants.GenericConstants.USER_CANCEL:
                response.<String>put(Response.KEY_STATUS_CODE, Constants.GenericConstants.USER_CANCEL);
                break;
            case Constants.GenericConstants.TRANSACTION_FAILED_IN_UPI_APP:
                response.<String>put(Response.KEY_STATUS_CODE, Constants.GenericConstants.TRANSACTION_FAILED_IN_UPI_APP);
                break;
            case Constants.GenericConstants.FAILED_TO_CONNECT_TO_SERVER:
                response.<String>put(Response.KEY_STATUS_CODE, Constants.GenericConstants.FAILED_TO_CONNECT_TO_SERVER);
                break;
        }

        return response;
    }

    public String getPackageSignature() {
        try {
            if(this.dataObjectFactoryCache.contains(SIGNATURE_DIGEST)) {
                return this.dataObjectFactoryCache.get(SIGNATURE_DIGEST);
            }
            PackageInfo packageInfo = getApplicationContext().getPackageManager().getPackageInfo(
                    getApplicationContext().getPackageName(), PackageManager.GET_SIGNATURES);
            Signature signature = packageInfo.signatures[0];
            byte[] sig = signature.toByteArray();
            MessageDigest digest = MessageDigest.getInstance("SHA-1");
            digest.update(sig);
            byte[] hashtext = digest.digest();
            String base64signature = Base64.encodeToString(hashtext, Base64.NO_WRAP);
            this.dataObjectFactoryCache.add(SIGNATURE_DIGEST, base64signature);
            return base64signature;
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
    }

    public static class InitializationBundle extends HashMap<String, Object> implements ObjectFactoryInitializationStrategy {

        @Override
        public void init(ObjectFactory objectFactory, InitializationBundle initializationBundle) {

        }

        @Override
        public boolean isCachingAllowed() {
            return false;
        }

        public <T> T get(String key, T defaultValue) {

            if (super.containsKey(key)) {
                return (T) super.get(key);
            }

            return defaultValue;
        }
    }

    //*********************************************************************
    // End of class
    //*********************************************************************
}
