package com.phonepe.intent.sdk.bridges;

import android.webkit.JavascriptInterface;

import com.phonepe.intent.sdk.contracts.iBridgeCallback;
import com.phonepe.intent.sdk.core.DataConfig;
import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.core.ObjectFactoryInitializationStrategy;
import com.phonepe.intent.sdk.models.Body;
import com.phonepe.intent.sdk.models.Error;
import com.phonepe.intent.sdk.models.Response;
import com.phonepe.intent.sdk.models.Value;
import com.phonepe.intent.sdk.ui.BaseWebActivity;
import com.phonepe.intent.sdk.utils.Constants;
import com.phonepe.intent.sdk.utils.SdkLogger;

/**
 * @author Suman PPE
 * @since 17/05/17.
 */
public class DataStore implements ObjectFactoryInitializationStrategy {

    public static final String TAG = "NativeStore";
    private DataConfig dataConfig;
    private iBridgeCallback bridgeCallback;
    private ObjectFactory objectFactory;

    private <T> void doCallback(String context, String key, String callback, T val) {


        Error error = null;
        Response response = null;
        Body<Value<T>> tBody = this.objectFactory.<Body>get(Body.class);

        if (val != null) {
            Value<T> tValue = this.objectFactory.<Value>get(Value.class);
            tValue.setKey(key);
            tValue.setValue(val);
            response = this.objectFactory.getResponse(Constants.GenericConstants.SUCCESS);
            tBody.setData(tValue);
        } else {
            error = this.objectFactory.<Error>get(Error.class);
            error.setCode(Constants.ErrorCodes.KEY_NOT_FOUND_ERROR);
            error.setMessage(Constants.ErrorCodes.KEY_NOT_FOUND_MESSAGE);
        }

        SdkLogger.v(TAG, String.format("onBridgeCallback triggered with callback = {%s}, error = {%s}, response = {%s}, context = {%s}, body = {%s}.", callback, error == null ? "null" : error.toJsonString(),
                response == null ? "null" : response.toJsonString(), context,
                tBody.toJsonString()));

        this.bridgeCallback.onBridgeCallBack(callback,
                error == null ? null : error.toJsonString(),
                response == null ? null : response.toJsonString(), context,
                tBody.toJsonString());

    }

    @JavascriptInterface
    public void getString(String context, String data, String callback) {

        SdkLogger.v(TAG, String.format("getString is called with parameters context = {%s}, data = {%s}, callback = {%s}.", context, data, callback));
        Value value = Value.fromJsonString(data, this.objectFactory, Value.class);
        String key = value.<String>get(Value.KEY);

        if (this.dataConfig.contains(key)) {

            this.<String>doCallback(context, key, callback, this.dataConfig.getString(key, ""));
        } else {

            this.<String>doCallback(context, key, callback, null);
        }

    }

    @JavascriptInterface
    public void getInt(String context, String data, String callback) {

        SdkLogger.v(TAG, String.format("getInt is called with parameters context = {%s}, data = {%s}, callback = {%s}.", context, data, callback));
        Value value = Value.fromJsonString(data, this.objectFactory, Value.class);
        String key = value.<String>get(Value.KEY);

        if (this.dataConfig.contains(key)) {

            this.<Integer>doCallback(context, key, callback, this.dataConfig.getInt(key, 0));
        } else {

            this.<Integer>doCallback(context, key, callback, null);
        }
    }

    @JavascriptInterface
    public void getBool(String context, String data, String callback) {

        SdkLogger.v(TAG, String.format("getBool is called with parameters context = {%s}, data = {%s}, callback = {%s}.", context, data, callback));
        Value value = Value.fromJsonString(data, this.objectFactory, Value.class);
        String key = value.<String>get(Value.KEY);

        if (this.dataConfig.contains(key)) {

            this.<Boolean>doCallback(context, key, callback, this.dataConfig.getBool(key, false));
        } else {

            this.<Boolean>doCallback(context, key, callback, null);
        }
    }

    @JavascriptInterface
    public void getFloat(String context, String data, String callback) {

        SdkLogger.v(TAG, String.format("getFloat is called with parameters context = {%s}, data = {%s}, callback = {%s}.", context, data, callback));
        Value value = Value.fromJsonString(data, this.objectFactory, Value.class);
        String key = value.<String>get(Value.KEY);

        if (this.dataConfig.contains(key)) {

            this.<Float>doCallback(context, key, callback, this.dataConfig.getFloat(key, -1f));
        } else {

            this.<Float>doCallback(context, key, callback, null);
        }
    }

    @JavascriptInterface
    public void setInt(String key, int value) {
        SdkLogger.v(TAG, String.format("setInt is called with parameters key = {%s}, value = {%s}.", key, value));
        this.dataConfig.saveInteger(key, value);
    }

    @JavascriptInterface
    public void setBool(String key, boolean value) {
        SdkLogger.v(TAG, String.format("setBool is called with parameters key = {%s}, value = {%s}.", key, value));
        this.dataConfig.saveBool(key, value);
    }

    @JavascriptInterface
    public void setString(String key, String value) {
        SdkLogger.v(TAG, String.format("setString is called with parameters key = {%s}, value = {%s}.", key, value));
        this.dataConfig.saveString(key, value);
    }

    @JavascriptInterface
    public void setFloat(String key, float value) {
        SdkLogger.v(TAG, String.format("setFloat is called with parameters key = {%s}, value = {%s}.", key, value));
        this.dataConfig.saveFloat(key, value);
    }

    @JavascriptInterface
    public void removeKey(String key) {
        SdkLogger.v(TAG, String.format("removeKey is called with parameters key = {%s}.", key));
        this.dataConfig.remove(key);
    }

    @JavascriptInterface
    public void clearDataStore() {
        SdkLogger.v(TAG, "clearDataStore is called to clear all data");
        this.dataConfig.clearAllData();
    }

    @Override
    public void init(ObjectFactory objectFactory, ObjectFactory.InitializationBundle initializationBundle) {

        this.objectFactory = objectFactory;
        this.dataConfig = objectFactory.<DataConfig>get(DataConfig.class);
        this.bridgeCallback = initializationBundle.<iBridgeCallback>get(BaseWebActivity.BRIDGE_CALLBACK, null);
    }

    @Override
    public boolean isCachingAllowed() {
        return false;
    }
}
