package com.phonepe.intent.sdk.api;

import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.phonepe.intent.sdk.core.ObjectFactory;
import com.phonepe.intent.sdk.networking.NetworkConstants;
import com.phonepe.intent.sdk.utils.AnalyticsManager;
import com.phonepe.intent.sdk.utils.SdkLogger;
import com.phonepe.intent.sdk.utils.Utils;

import java.util.HashMap;
import java.util.Map;

import static com.phonepe.intent.sdk.utils.AnalyticsManager.Events.SDK_TRANSACTION_REQUEST_CREATED;

/**
 * @author Sharath Pandeshwar
 * @since 23/09/16.
 */
public class TransactionRequest implements Parcelable {


    /**
     * Checksum for the request generated as per the documentation.
     */
    private String checksum;

    /**
     * API version as returned by server
     */
    private String apiUrl;
    private String data;
    private String redirectUrl;
    private Map<String, String> headers;

    private TransactionRequest() {

        this.headers = new HashMap<>();
    }


    public Map<String, String> getHeaderMap() {

        this.headers.put(NetworkConstants.KEY_X_VERIFY, checksum);
        return this.headers;
    }

    public String getAPIUrl() {
        return apiUrl;
    }


    public boolean isDebitRequest() {
        return apiUrl.contains("debit");
    }

    public String getChecksum() {
        return checksum;
    }

    public String getData() {
        return data;
    }

    public String getRedirectUrl() {
        return redirectUrl;
    }

    @Override
    public String toString() {
        return "TransactionRequest{" +
                "checksum='" + this.checksum + '\'' +
                ", apiUrl='" + this.apiUrl + '\'' +
                ", data='" + this.data + '\'' +
                ", redirectUrl='" + this.redirectUrl + '\'' +
                ", headers=" + headers +
                '}';
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.checksum);
        dest.writeString(this.apiUrl);
        dest.writeString(this.data);
        dest.writeString(this.redirectUrl);
        dest.writeMap(this.headers);
    }

    protected TransactionRequest(Parcel in) {
        this.checksum = in.readString();
        this.apiUrl = in.readString();
        this.data = in.readString();
        this.redirectUrl = in.readString();
        this.headers = in.readHashMap(TransactionRequest.class.getClassLoader());
    }

    public static final Creator<TransactionRequest> CREATOR = new Creator<TransactionRequest>() {
        @Override
        public TransactionRequest createFromParcel(Parcel in) {
            return new TransactionRequest(in);
        }

        @Override
        public TransactionRequest[] newArray(int size) {
            return new TransactionRequest[size];
        }
    };

    public static class TransactionRequestBuilder {

        private static final String TAG = "TrxRequestBuilder";
        private String checksum;
        private String apiUrl;
        private String data;
        private String redirectUrl;
        private HashMap<String, String> headers;


        public TransactionRequestBuilder setChecksum(@NonNull String checksum) {
            this.checksum = checksum;
            return this;
        }

        public TransactionRequestBuilder setUrl(@NonNull String apiUrl) {
            this.apiUrl = apiUrl;
            return this;
        }

        public TransactionRequestBuilder setData(@NonNull String data) {
            this.data = data;
            return this;
        }

        public TransactionRequestBuilder setRedirectUrl(@NonNull String redirectUrl) {
            this.redirectUrl = redirectUrl;
            return this;
        }


        public TransactionRequestBuilder setHeaders(@NonNull HashMap<String, String> headers) {
            this.headers = headers;
            return this;
        }

        public TransactionRequest build() {


            /* Not using external method lest I will have to mock them too */
            if (Utils.isNullOrEmpty(this.data)) {
                SdkLogger.e(TAG, "Setting data is mandatory.");
                throw new RuntimeException("Setting data is mandatory.");
            }

            if (Utils.isNullOrEmpty(this.checksum)) {
                SdkLogger.e(TAG, "Setting checksum is mandatory.");
                throw new RuntimeException("Setting checksum is mandatory.");
            }

            if (Utils.isNullOrEmpty(this.apiUrl)) {
                SdkLogger.e(TAG, "Setting Url is mandatory.");
                throw new RuntimeException("Setting Url is mandatory.");
            }

            TransactionRequest transactionRequest = new TransactionRequest();
            transactionRequest.data = this.data;
            transactionRequest.checksum = this.checksum;
            transactionRequest.apiUrl = this.apiUrl;
            transactionRequest.redirectUrl = this.redirectUrl;
            Utils.cacheTransactionId(transactionRequest.data);
            if (this.headers != null && !this.headers.isEmpty()) {
                transactionRequest.headers.putAll(this.headers);
            }

            ObjectFactory objectFactory = null;
            try {
                objectFactory = PhonePe.getObjectFactory();
                AnalyticsManager analyticsManager = objectFactory.<AnalyticsManager>get(AnalyticsManager.class);
                analyticsManager.submit(analyticsManager.getEvent(SDK_TRANSACTION_REQUEST_CREATED));
            } catch (PhonePeInitException e) {
                SdkLogger.e(TAG, e.getMessage(), e);
            }
            return transactionRequest;
        }
    }
}
