(ns paren.serene.compiler
  (:refer-clojure :exclude [compile])
  (:require
   [clojure.spec.alpha :as s]
   [clojure.string :as str]
   [clojure.walk :as walk]
   [paren.serene.introspection :as introspection]
   #?(:cljs [cljs.reader :refer [read-string]]))
  #?(:cljs (:require-macros
            [paren.serene.compiler])))

(def ^:private tmp-ns
  "paren.serene.compiler.tmp")

(defn ^:private tmp-spec-name?
  [k]
  (and
    (keyword? k)
    (= (namespace k) tmp-ns)))

(defn ^:private tmp-spec-name
  [k-or-obj]
  (let [ks (if (keyword? k-or-obj)
             [k-or-obj]
             (loop [obj k-or-obj
                    ks ()]
               (if obj
                 (recur
                   (::parent obj)
                   (cons (:name obj) ks))
                 ks)))]
    (->> ks
      (mapcat (juxt namespace name))
      (remove nil?)
      (str/join ".")
      (keyword tmp-ns))))

(defn ^:private raw-spec-name
  [k]
  {:pre [(tmp-spec-name? k)]}
  (let [segs (str/split (name k) #"\.")
        ns  (when-let [ns-segs (butlast segs)]
              (str/join "." ns-segs))
        n (last segs)]
    (keyword ns n)))

(defn ^:private prefix-spec-name
  [prefix k]
  {:pre [(tmp-spec-name? k)]}
  (let [prefix (name prefix)
        segs (str/split (name k) #"\.")
        ns (->> (butlast segs)
             (cons prefix)
             (str/join "."))
        n (last segs)]
    (keyword ns n)))

(defn ^:private maybe-prefix-spec-name
  [prefix k]
  (if (tmp-spec-name? k)
    (prefix-spec-name prefix k)
    k))

(def ^:private assoc-specs-object-spec
  (s/or
    :EnumValue ::introspection/EnumValue
    :Field ::introspection/Field
    :InputValue ::introspection/InputValue
    :Type ::introspection/Type))

(defmulti ^:private -assoc-specs
  (fn [{:as obj
        :keys [__typename kind]}]
    {:pre [(s/valid? assoc-specs-object-spec obj)]}
    (case __typename
      :__EnumValue :ENUM_VALUE
      :__Field :FIELD
      :__InputValue :INPUT_VALUE
      :__Type kind)))

(defn ^:private assoc-specs
  ([obj]
   (assoc-specs nil obj))
  ([parent obj]
   (cond
     (s/valid? assoc-specs-object-spec obj) (-assoc-specs (assoc obj ::parent parent))
     (s/valid? ::introspection/Directive obj) obj
     (coll? obj) (mapv assoc-specs obj)
     :else obj)))

(defn ^:private assoc-child-specs
  [obj k]
  (->> obj
    (partial assoc-specs)
    (partial mapv)
    (update obj k)))

(defn ^:private type-ref-spec
  [type-ref]
  (->> type-ref
    (walk/postwalk (fn [x]
                     (if (map? x)
                       (let [{:keys [kind name ofType]} x]
                         (if (= kind :NON_NULL)
                           (assoc ofType :non-null? true)
                           x))
                       x)))
    ((fn type-ref->spec [{:keys [non-null? kind name ofType]}]
       (let [spec (if (= kind :LIST)
                    `(s/coll-of ~(type-ref->spec ofType)
                       :kind sequential?)
                    (tmp-spec-name name))]
         (if non-null?
           spec
           `(s/or ; Avoid using `s/nilable` due to https://dev.clojure.org/jira/browse/CLJS-2940
              :null nil?
              :non-null ~spec)))))))

(defn ^:private input-values-spec
  [input-values]
  (->> input-values
    (reduce
      (fn [m {:keys [defaultValue type]
              ::keys [spec]}]
        (let [opt? (or
                     defaultValue
                     (not= (:kind type) :NON_NULL))
              k (if opt? :opt-un :req-un)]
          (update m k conj (::name spec))))
      {:opt-un []
       :req-un []})
    (apply concat)
    (cons `s/keys)))

(defmethod -assoc-specs :ENUM [obj]
  (let [obj (assoc-child-specs obj :enumValues)]
    (assoc obj ::spec {::name (tmp-spec-name obj)
                       ::form (->> obj
                                :enumValues
                                (mapcat (juxt :name (comp ::name ::spec)))
                                (cons `s/or))})))

(defmethod -assoc-specs :ENUM_VALUE [obj]
  (assoc obj ::spec {::name (tmp-spec-name obj)
                     ::form #{(-> obj :name name)}}))

(defmethod -assoc-specs :FIELD [obj]
  (let [obj (assoc-child-specs obj :args)]
    (-> obj
      (assoc ::spec {::name (tmp-spec-name obj)
                     ::form (if (= (:name obj) :__typename)
                              #{(-> obj ::parent :name name)}
                              (-> obj :type type-ref-spec))})
      (assoc ::args-spec {::name (tmp-spec-name {::parent obj :name :%})
                          ::form (-> obj :args input-values-spec)}))))

(defmethod -assoc-specs :INPUT_OBJECT [obj]
  (let [obj (assoc-child-specs obj :inputFields)]
    (assoc obj ::spec {::name (tmp-spec-name obj)
                       ::form (->> obj
                                :inputFields
                                input-values-spec)})))

(defmethod -assoc-specs :INPUT_VALUE [obj]
  (assoc obj ::spec {::name (tmp-spec-name obj)
                     ::form (-> obj :type type-ref-spec)}))

(defmethod -assoc-specs :INTERFACE [obj]
  (let [obj (assoc-child-specs obj :fields)]
    (assoc obj ::spec {::name (tmp-spec-name obj)
                       ::form (->> obj
                                :possibleTypes
                                (mapcat (juxt :name tmp-spec-name))
                                (cons `s/or)
                                doall)})))

(defmethod -assoc-specs :OBJECT [obj]
  (let [obj (-> obj
              (update :fields conj {:__typename :__Field
                                    :args []
                                    :deprecationReason nil
                                    :description nil
                                    :isDeprecated false
                                    :name :__typename
                                    :type {:__typename :__Type
                                           :kind :NON_NULL
                                           :name nil
                                           :ofType  {:__typename :__Type
                                                     :kind :SCALAR
                                                     :name :String
                                                     :ofType nil}}})
              (assoc-child-specs :fields))]
    (assoc obj ::spec {::name (tmp-spec-name obj)
                       ::form (->> obj
                                :fields
                                (mapv (comp ::name ::spec))
                                (list :opt-un)
                                (cons `s/keys))})))

(defmethod -assoc-specs :SCALAR [obj]
  (assoc obj ::spec {::name (tmp-spec-name obj)
                     ::form (case (:name obj)
                              :Boolean `boolean?
                              :Float `float?
                              :ID `string?
                              :Int `integer?
                              :String `string?
                              `any?)}))

(defmethod -assoc-specs :UNION  [obj]
  (assoc obj ::spec {::name (tmp-spec-name obj)
                     ::form (->> obj
                              :possibleTypes
                              (mapcat (juxt :name tmp-spec-name))
                              (cons `s/or)
                              doall)}))

(defn ^:private collect-specs
  [resp]
  (->> resp
    (tree-seq coll? seq)
    (filter map?)
    (map (juxt ::name ::form))
    (filter (partial every? some?))
    (reduce
      (fn [m [k v]]
        (when (contains? m k)
          (throw (ex-info
                   (str "Duplicate schema spec name: " (raw-spec-name k))
                   {:spec (raw-spec-name k)})))
        (assoc m k v))
      {})))

(defn ^:private nonconform-specs
  [tmp-spec-map]
  (->> tmp-spec-map
    (map (fn [[k v]]
           [k
            (if (seq? v)
              `(s/nonconforming ~v)
              v)]))
    (into {})))

(defn ^:private extend-specs
  [extend-fn tmp-spec-map]
  (reduce
    (fn [spec-map tmp-k]
      (if-some [ext (-> tmp-k raw-spec-name extend-fn)]
        (update spec-map tmp-k (fn [spec]
                                 `(s/and ~spec ~ext)))
        spec-map))
    tmp-spec-map
    (keys tmp-spec-map)))

(defn ^:private alias-specs
  [alias-fn tmp-spec-map]
  (reduce
    (fn [spec-map tmp-k]
      (let [raw-k (raw-spec-name tmp-k)
            aliases (alias-fn raw-k)]
        (reduce
          (fn [spec-map alias]
            (when (contains? spec-map alias)
              (throw (ex-info (str "Alias not unique: " alias) {:alias alias})))
            (assoc spec-map alias tmp-k))
          spec-map
          aliases)))
    tmp-spec-map
    (keys tmp-spec-map)))

(defn ^:private prefix-specs
  [prefix tmp-spec-map]
  (->> tmp-spec-map
    (map (fn [[k v]]
           [k (walk/postwalk (partial maybe-prefix-spec-name prefix) v)]))
    (into {})
    (reduce
      (fn [m [k v]]
        (let [k (maybe-prefix-spec-name prefix k)]
          (when (contains? m k)
            (throw (ex-info (str "Duplicate prefixed spec name: " k) {:spec k})))
          (assoc m k v)))
      {})))

(defn ^:private topo-sort-specs
  [prefix-spec-map]
  (let [sort-aliases (fn sort-aliases [sorted unsorted]
                       (let [no-deps (->> unsorted
                                       (remove (fn [[k v]]
                                                 (contains? unsorted v)))
                                       (into {}))
                             sorted' (concat sorted no-deps)
                             unsorted' (apply dissoc unsorted (keys no-deps))]
                         (if (seq unsorted')
                           (sort-aliases sorted' unsorted')
                           sorted')))
        aliases (->> prefix-spec-map
                  (filter #(-> % val keyword?))
                  (into {}))]
    (concat
      (apply dissoc prefix-spec-map (keys aliases))
      (sort-aliases [] aliases))))

(defn ^:private def-specs
  [spec-entries]
  (map
    (fn [[k v]]
      `(s/def ~k ~v))
    spec-entries))

(defn ^:private conform!
  [spec x]
  (let [conformed (s/conform spec x)]
    (if (s/invalid? conformed)
      (throw (ex-info
               (str "Invalid " spec ": " (s/explain-str spec x))
               {:spec spec
                :invalid x}))
      conformed)))

(s/def ::aliases (s/and
                   (s/or
                     :coll (s/nilable
                             (s/coll-of qualified-keyword?
                               :distinct true))
                     :keyword qualified-keyword?)
                   (s/conformer
                     (fn [[k v]]
                       (case k
                         :coll (set v)
                         :keyword #{v})))))

(s/def ::alias (s/and
                 ifn?
                 (s/conformer
                   (fn [alias-fn]
                     (fn alias-wrapper [kw]
                       (conform! ::aliases (alias-fn kw)))))))

(defn ^:private serializable?
  [form]
  (try
    (= (-> form pr-str read-string) form)
    (catch #?(:clj Throwable :cljs :default) e
      false)))

(s/def ::extension (s/nilable
                     (s/nonconforming
                       (s/or
                         :keyword qualified-keyword?
                         :seq (s/and seq? serializable?)
                         :symbol symbol?))))

(s/def ::extend (s/and
                  ifn?
                  (s/conformer
                    (fn [extend-fn]
                      (fn extend-wrapper [kw]
                        (conform! ::extension (extend-fn kw)))))))

(s/def ::prefix (s/and
                  (s/or
                    :namespace (fn namespace? [x]
                                 (instance?
                                   #?(:clj clojure.lang.Namespace
                                      :cljs cljs.core.Namespace)
                                   x))
                    :ident simple-ident?)
                  (s/conformer
                    (fn [[k v]]
                      (case k
                        :namespace (ns-name v)
                        :ident v)))))

(defn compile
  "Takes a GraphQL introspection query response.
  Returns a topologically sorted vector of `s/def` forms."
  ([response]
   (compile response {}))
  ([response {:keys [alias
                     extend
                     prefix]
              :or {alias (constantly #{})
                   extend (constantly nil)
                   prefix *ns*}}]
   (->> (conform! ::introspection/response response)
     assoc-specs
     collect-specs
     nonconform-specs
     (extend-specs (conform! ::extend extend))
     (alias-specs (conform! ::alias alias))
     (prefix-specs (conform! ::prefix prefix))
     topo-sort-specs
     def-specs
     vec)))
