;;; ## Action Plan Building
  (defn action-plan
    "Build the action plan for the specified `plan-fn` on the given `node`, within
  the context of the `service-state`. The `plan-state` contains all the
  settings, etc, for all groups. `target-map` is a map for the session
  describing the target."
    [service-state environment plan-fn args target-map]
    {:pre [(not (map? plan-fn)) (fn? plan-fn)
           (map? target-map)
           (or (nil? environment) (map? environment))]}
    (fn action-plan [plan-state]
      (tracef "action-plan plan-state %s" plan-state)
      (let [s (with-session
                (add-session-verification-key
                 (merge
                  {:user (:user environment)}
                  target-map
                  {:service-state service-state
                   :plan-state plan-state
                   :environment environment}))
                (apply plan-fn args)
                (check-session (session) '(plan-fn))
                (session))]
        (let [[action-plan session] (get-session-action-plan s)
              [action-plan session] (translate action-plan session)]
          [action-plan (:plan-state session)])))))

(defn phase-args [phase]
  (if (keyword? phase)
    nil
    (rest phase)))

(defn- phase-kw [phase]
  (if (keyword? phase)
    phase
    (first phase)))

(defn target-phase [target phase]
  (-> target :phases (get (phase-kw phase))))

(defmulti target-action-plan
  "Build action plans for the specified `phase` on all nodes or groups in the
  given `target`, within the context of the `service-state`. The `plan-state`
  contains all the settings, etc, for all groups."
  (fn [service-state plan-state environment phase target]
    (tracef "target-action-plan %s" (:target-type target :node))
    (:target-type target :node)))

(defmethod target-action-plan :node
  [service-state plan-state environment phase target]
  {:pre [target (:node target)]}
  (fn [plan-state]
    (logutils/with-context [:target (-> target :node primary-ip)]
      (with-script-for-node target plan-state
        ((action-plan
          service-state environment
          (target-phase target phase) (phase-args phase)
          {:server target})
         plan-state)))))

(defmethod target-action-plan :group
  [service-state plan-state environment phase group]
  {:pre [group]}
  (fn [plan-state]
    (logutils/with-context [:target (-> group :group-name)]
      ((action-plan
        service-state environment
        (target-phase group phase) (phase-args phase)
        {:group group})
       plan-state))))

(defn action-plans
  [service-state plan-state environment phase targets]
  (let [targets-with-phase (filter #(target-phase % phase) targets)]
    (tracef
     "action-plans: phase %s targets %s targets-with-phase %s"
     phase (vec targets) (vec targets-with-phase))
    (with-monad state-m
      (domonad
       [action-plans
        (m-map
         #(target-action-plan service-state plan-state environment phase %)
         targets-with-phase)]
       (map
        #(hash-map :target %1 :phase (phase-kw phase) :action-plan %2)
        targets-with-phase action-plans)))))


(defn execute-action-plan*
  "Execute the `action-plan` on the `target`."
  [session executor execute-status-fn
   {:keys [action-plan phase target-type target]}]
  (tracef "execute-action-plan*")
  (with-session session
    (let [[result session] (execute
                            action-plan session executor execute-status-fn)]
      {:target target
       :target-type target-type
       :plan-state (:plan-state session)
       :result result
       :phase (phase-kw phase)})))

(defmulti execute-action-plan
  "Execute the `action-plan` on the `target`."
  (fn [service-state plan-state environment user executor execute-status-fn
       {:keys [action-plan phase target]}]
    (:target-type target :node)))

(defmethod execute-action-plan :node
  [service-state plan-state environment user executor execute-status-fn
   {:keys [action-plan phase target-type target] :as action-plan-map}]
  (tracef "execute-action-plan :node")
  (logutils/with-context [:target (-> target :node primary-ip)]
    (with-script-for-node target plan-state
      (execute-action-plan*
       {:server target
        :service-state service-state
        :plan-state plan-state
        :user user
        :environment environment}
       executor execute-status-fn action-plan-map))))

(defmethod execute-action-plan :group
  [service-state plan-state environment user executor execute-status-fn
   {:keys [action-plan phase target-type target] :as action-plan-map}]
  (tracef "execute-action-plan :group")
  (logutils/with-context [:target (-> target :group-name)]
    (execute-action-plan*
     {:group target
      :service-state service-state
      :plan-state plan-state
      :user user
      :environment environment}
     executor execute-status-fn action-plan-map)))
