(ns pallet.compute.jclouds.ec2-groups
  "Implementation of groups on ec2"
  (:use
   [org.jclouds.ec2.security-group2 :as [sg2]]))


;;; missing from security-group2
(defn authorize-group
  "Adds permissions to a security group.

   e.g. (authorize compute \"jclouds#webserver#us-east-1\" 80 :ip-range \"0.0.0.0/0\")
        (authorize compute \"jclouds#webserver#us-east-1\" [1000,2000] :protocol :udp)"

  [compute group-name port & {:keys [protocol ip-range region]}]
  (let [group ((groups compute :region region) group-name)
        [from-port to-port] (if (number? port) [port port] port)]
    (if group
      (.authorizeSecurityGroupIngressInRegion
       (sg-service compute) (ebs/get-region region) (.getName group) (get-protocol protocol) from-port to-port (or ip-range "0.0.0.0/0"))
      (throw (IllegalArgumentException.
              (str "Can't find security group for name " group-name))))))


(defn )


(defprotocol SecurityGroupProvider
  "Allows a configurable implementation of security groups"
  (security-group [_ net-security-spec]))


(defaction create-security-group
  [{:keys [group-name allow deny description region]}])

(defaction delete-security-group
  [{:keys [group-name region]}])

(defimplementation create-security-group
  [session {:keys [group-name allow deny description region]}]
  (create-group
   (or
    (node/compute-service (target-node session))
    (:compute session))
   group-name
   (into {} (filter val {:description description :region region}))))

(defimplementation delete-security-group
  [session {:keys [group-name region]}]
  (delete-group
   (or
    (node/compute-service (target-node session))
    (:compute session))
   group-name
   (into {} (filter val {:region region}))))

(defimplementation authorize-security-group
  [session compute group-name port & {:keys [protocol ip-range region]}]
  )


(defn ec2-security-group
  [{:keys [allow deny] :as security-spec}]
  (server-spec
   :phases {:settings (plan-fn
                        ;; record map of which groups allow what
                        )
            :create-group (plan-fn
                            [group-name target-group
                             roles target-roles]
                            (create-security-group group-name)
                            (m-map declare-security-group roles))
            :destroy-group (plan-fn
                             [group-name target-group
                              roles target-roles]
                             (remove-security-group group-name)
                             (map maybe-remove-security-group roles))
            :group-configure (plan-fn
                               ;; act on declaration of which groups allow what
                               )}))

(deftype JCloudsEc2SecurityGroupProvider
    SecurityGroupProvider
  (security-group [_ security-spec] (ec2-security-group security-spec)))




(comment
  ;; iptables equivalent
  (defaction declare-security-group
    [group-name])

  (defaction create-security-group
    [group-name])

  (defimplementation
    [session {:keys [allow deny]}])


  (defn iptables-security-group
    [{:keys [allow deny] :as security-spec}]
    (server-spec
     :phases {:settings (plan-fn
                          ;; record map of which groups allow what
                          )
              :create-group (plan-fn
                              [group-name target-group
                               roles target-roles]
                              (create-security-group group-name)
                              (m-map declare-security-group roles))
              :destroy-group (plan-fn
                               [group-name target-group
                                roles target-roles]
                               (remove-security-group group-name)
                               (map maybe-remove-security-group roles))
              :group-configure (plan-fn
                                 ;; act on declaration of which groups allow what
                                 )}))

  (deftype IpTablesSecurityGroupProvider
      SecurityGroupProvider
    (security-group [_ security-spec] (iptables-security-group security-spec))))
