(ns com.palletops.leaven
  "A component composition library."
       
           
                                                 
                                                    
                                        
        
  (:require-macros
   [com.palletops.api-builder.api :refer [defn-api]]
   [schema.macros :refer [=>]])
        
  (:require
   [com.palletops.leaven.protocols :as protocols]
   [schema.core :as schema]))

(defn-api start
  "Start a component."
  {:sig [[schema/Any :- schema/Any]]}
  [component]
  (if (protocols/startable? component)
    (protocols/start component)
    component))

(defn-api stop
  "Stop a component."
  {:sig [[schema/Any :- schema/Any]]}
  [component]
  (if (protocols/stoppable? component)
    (protocols/stop component)
    component))

(defn-api status
  "Ask a component for its status."
  {:sig [[schema/Any :- schema/Any]]}
  [component]
  (if (protocols/queryable? component)
    (protocols/status component)))

(defn-api startable?
  "Predicate for testing whether `x` satisfies the Startable protocol."
  {:sig [[schema/Any :- schema/Any]]}
  [x]
  (protocols/startable? x))

(defn-api stoppable?
  "Predicate for testing whether `x` satisfies the Stoppable protocol."
  {:sig [[schema/Any :- schema/Any]]}
  [x]
  (protocols/stoppable? x))

(defn-api queryable?
  "Predicate for testing whether `x` satisfies the Queryable protocol."
  {:sig [[schema/Any :- schema/Any]]}
  [x]
  (protocols/queryable? x))

(defn ^:internal apply-components
  "Execute a function on a sequence of components from a record.
  Exceptions are caught and propagate with a `:system` data element
  that contains the partially updated system component, a `:component`
  key that is the keyword for the component that caused the exception,
  and `:incomplete` which is a sequence of keywords for the components
  where the operation was not completed."
  [f rec sub-components operation-name on-f]
  (loop [rec rec cs sub-components]
    (if-let [c (first cs)]
      (let [post-f (get on-f c)
            res (try
                  (-> (update-in rec [c] f)
                      (cond-> post-f (post-f c)))
                  (catch        js/Error                 e
                         (let [d (ex-data e)]
                           (throw
                            (ex-info
                             (str "Exception while " operation-name
                                  " " c
                                  " system sub-component.")
                             (merge
                              {:system rec
                               :component c
                               :sub-components sub-components
                               :completed (subvec sub-components
                                                  0 (- (count sub-components)
                                                       (count cs)))
                               :uncompleted cs
                               :operation-name operation-name
                               :type ::system-failed}
                              (if (= ::system-failed (:type d))
                                ;; ensure we report the most nested details
                                d))
                             e)))))]
        (recur res (rest cs)))
      rec)))

(defn-api update-components
  "Returns a function to update the system components given by
  `component-specs`.


  , with a sub-component, assuming the component acts like a map.

  Can be used as a value in an :on-start option map in defrecord to
  get components updated with their started values."
  {:sig [[(schema/either
           [schema/Keyword]
           {schema/Keyword schema/Keyword})
          :- (=> schema/Any schema/Any schema/Keyword)]]}
  [component-specs]
  (fn [component sub-kw]
    (reduce
     #(cond
       (keyword? %2) (assoc-in %1 [%2 sub-kw] (sub-kw component))
       :else (assoc-in %1 [(key %2) (val %2)] (sub-kw component)))
     component
     component-specs)))

(defn comp-on-f
  "Compose two functions for use with an on-start or on-finish
  function."
  [f1 f2]
  (fn [c kw]
    (-> c
        (f1 :channel)
        (f2 :channel))))

     
                
                                      
           
         
                     
                    
              
                         
                                   
                 
              
                    
                                   
                   
     
            

     
                     
                                                                       
                                 
         
                  
                                     
              
                                     
                                       
                            
                         


     
                         
                                                                    
                                                               
                                                                      
                                                                  
                                                                     
                                      

                                                               

                                                                   
                                                                    
                                                                     
                                                                 
                                                               
                                                                      
                                                                      
             

           
                            
                                     

                                                                   
                                                                  
                                                              
                                  

            
                                                
                                                       

                                                             
                                                                     
         

                                                                 
                           
                                 
                                                           
                                                           
                                                 
                                        
                                                          
                        
                          
                    
                                                                             
        
                                                                                  
                                                 
               
                            
                              
                            
                                                                                
                            
                             
                            
                                                                            
                            
                               
                            
                                                                      

;;;;;;;;;;;; This file autogenerated from src/cljx/com/palletops/leaven.cljx
