(ns com.omarpolo.robotto.core
  (:require [com.omarpolo.robotto.interceptor :as interceptor]
            [com.omarpolo.robotto.effect :as effect]
            [org.httpkit.client :as client]
            [clojure.data.json :as json]
            [clojure.core.async :as async :refer [go chan >! <! >!! <!! put! alts!!]]))

(defn- update-whole
  "Like update, but pass the whole map insteaf of only `(k m)` to the
  function."
  [m k func]
  (assoc m k (func m)))

(defn- parse-json [str]
  (try
    (if str
      (json/read-str str :key-fn keyword))
    (catch Throwable e
      (println "cannot parse" str "due to" e)
      nil)))

(defn- to-json [x]
  (json/write-str x))



(defn- method-url [{url :req-url} method-name]
  (str url "/" method-name))

(defn- make-request
  [ctx {:keys [name params]} callback]
  (try
    (client/post (method-url ctx name)
                 {:headers {"Content-Type" "application/json"
                            "Accept" "application/json"}
                  :body (to-json params)}
                 (fn [{:keys [body error]}]
                   (callback {:response (-> body parse-json :data :result)
                              :error error})))
    (catch Throwable e
      (callback {:response nil
                 :error e}))))

(defn- update-type [update]
  (cond
    (:message update)
    ::message

    (:callback_query update)
    ::callback-query

    :else
    ::unknown))

(defn- extract-command
  "Extract the command name from the message."
  [{:keys [offset length]} {:keys [text]}]
  (subs text (inc offset) (+ offset length)))

(defn- is-command? [{entities :entities, :as message}]
  (loop [entities entities]
    (let [entity (first entities)
          t (:type entity)]
      (cond
        (empty? entities) nil
        (= "bot_command" t) (keyword (extract-command entity message))
        :else (recur (rest entities))))))

(defn- chain-for-update [{:keys [error command text msg callback-query]}
                         {:keys [message], :as update}]
  (case (update-type update)
    ::message (if-let [cmd (is-command? message)]
                {:chain (cmd command)
                 :data  message}
                (or (first (for [k '(:text :new_chat_members)
                                 :when (k message)]
                             {:chain (k msg)
                              :data message}))
                    {:chain error
                     :data {:msg "unknown message type"
                            :type ::unknown-message
                            :data message}}))

    ::callback-query {:chain callback-query
                      :data (:callback_query message)}

    ::unknown {:chain error
               :data {:msg "unknown update type"
                      :type ::unknown-update
                      :data update}}))

(defn- notify
  "Notify an error by running the error chain."
  [{error :error, :as ctx} err]
  (interceptor/run error err)
  ctx)

(defn- consume-updates
  "Runs the action for the given updates, then returns a new context."
  [{error :error, :as ctx} updates]
  (doseq [update updates]
    (let [{:keys [chain data]} (chain-for-update ctx update)]
      (if (empty? chain)
        (interceptor/run error {:msg "missing action for update"
                                :type ::missing-action
                                :data update})
        (interceptor/run chain data))))

  ;; return a new context with the :update-offset updated
  (let [{u :update-offset} ctx]
    (if (empty? updates)
      ctx
      (assoc ctx :update-offset
             (inc (reduce #(max %1 (:update_id %2))
                          u updates))))))

(defn- realize-requests
  "Run all the requests."
  [ctx reqs out-c]
  (when reqs
    (make-request ctx (first reqs)
                  (fn [response]
                    (put! out-c response
                          (fn [_]
                            (realize-requests ctx (next reqs) out-c)))))))

(defn get-updates
  "Retrieve updates from telegram and process them, yielding back a new contex."
  [{:keys [update-offset timeout], ch ::chan, :as ctx}]
  (make-request ctx {:name   'getUpdates
                     :params {:offset          update-offset
                              :timeout         timeout
                              :allowed_updates ["message" "callback_query"]}}
                #(>!! ch [%1 %2]))
  (let [{:keys [response error]} (<!! ch)
        ctx                      (cond
                                   response (consume-updates ctx response)
                                   error    (notify ctx {:msg  "error during update fetching"
                                                         :type ::transport-error
                                                         :data error}))
        out-chan                 (chan 32)]
    (realize-requests ctx (::effect/reqs ctx) out-chan)
    ctx))



(def base-config
  "The default configuration."
  {:token nil
   :base-url "https://api.telegram.org/bot"
   :timeout 5
   :exit-chan nil ;; will be created a new chan during build
   :default-interceptors []})

(defn new-ctx
  "Build a new context."
  []
  base-config)

(defn set-token [ctx token]
  (assoc ctx :token token))

(defn set-base-url [ctx url]
  (assoc ctx :base-url url))

(defn set-timeout [ctx timeout]
  (assoc ctx :timeout timeout))

(defn on-new-chat-members [ctx i]
  (update-in ctx [:msg :new_chat_members] interceptor/chain i))

(defn on-command [ctx command i]
  (update-in ctx [:command command] interceptor/chain i))

(defn on-text [ctx i]
  (update-in ctx [:msg :text] interceptor/chain i))

(defn on-error [ctx i]
  (update ctx :error interceptor/chain i))

(defn on-callback-query [ctx i]
  (update ctx :callback-query interceptor/chain i))

(defn build-ctx
  "Builds the context."
  [{:keys [base-url token], :as ctx}]
  (-> ctx
      (assoc :req-url (str base-url token))
      (assoc :update-offset 0)
      (update-whole :me #(<!! (get-me %)))
      (assoc ::bus (chan 8))
      (assoc :exit-chan (chan))))
