(ns olauzon.accumulator
  (:require [clojure.core.async :refer [>!!
                                        alt!!
                                        chan
                                        close!
                                        put!
                                        thread
                                        timeout]]
            [taoensso.timbre :as log]))


(defn run!
 ([size msecs]
  (run! size msecs (chan) (chan) (chan)))
 ([size msecs e-in b-out c-in]
     "Buffer events accumulating from `e-in` for a maximum batch of `size` or
     `msecs` milliseconds. Batches are put in `b-out`."
    (let [s (dec size)]
      (thread
        (loop [batch []
               to    (timeout msecs)]
          (alt!!
            e-in ([e]
              (if (< (count batch) (dec size))
                (recur (conj batch e) to)
                (do
                  (>!! b-out (conj batch e))
                  (recur [] (timeout msecs)))))
            c-in ([_]
              (when-not (empty? batch) (>!! b-out batch)))
            to ([_]
              (if (empty? batch)
                (recur batch (timeout msecs))
                (do
                  (>!! b-out batch)
                  (recur [] (timeout msecs)))))))))
      {:e-in  e-in
       :b-out b-out
       :c-in  c-in
       :size  size
       :msecs msecs}))

;; pooling


;; https://github.com/clojure/core.incubator/blob/master/src/main/clojure/clojure/core/incubator.clj
(defn dissoc-in
  "Dissociates an entry from a nested associative structure returning a new
  nested structure. keys is a sequence of keys. Any empty maps that result
  will not be present in the new structure."
  [m [k & ks :as keys]]
  (if ks
    (if-let [nextmap (get m k)]
      (let [newmap (dissoc-in nextmap ks)]
        (if (seq newmap)
          (assoc m k newmap)
          (dissoc m k)))
      m)
    (dissoc m k)))

(def pools
  (atom {}))

(defn p-run!
  ([size msecs b-out p-key t-key ta]
    "Buffer events accumulating from `e-in` for a maximum batch of `size` or
    `msecs` milliseconds. Batches are put in `b-out`."
    (let [s (dec size)
          e-in (chan size)
          c-in (chan)
          c-out (chan)
          t (thread
              (log/info (str "starting: " p-key " " t-key))
              (loop [batch []
                     to    (timeout msecs)]
                (alt!!
                  e-in ([e]
                    (if (< (count batch) (dec size))
                      (recur (conj batch e) to)
                      (do
                        (log/info (str "size reached: " p-key " " t-key))
                        (>!! b-out (conj batch e))
                        (recur [] (timeout msecs)))))
                  c-in ([a]
                    (log/info (str "closing: " p-key " " t-key))
                    (swap! pools dissoc-in [p-key t-key])
                    (close! e-in)
                    (>!! c-out t-key)
                    (when-not (empty? batch) (>!! b-out batch)))
                  to ([_]
                    (log/info (str "timeout: " p-key " " t-key))
                    (if (empty? batch)
                      (do
                        (condp = ta
                          :close (>!! c-in :close)
                          nil)
                        (recur batch (timeout msecs)))
                      (do
                        (>!! b-out batch)
                        (recur [] (timeout msecs))))))))
          r {:e-in   e-in
             :b-out  b-out
             :c-in   c-in
             :c-out  c-out
             :p-key  p-key
             :t-key  t-key
             :size   size
             :msecs  msecs
             :ta     ta
             :thread t}]
      (swap! pools assoc-in [p-key t-key] r)
      r)))

(defn get!
  [size msecs b-out p-key t-key ta]
  (or (get-in @pools [p-key t-key])
      (p-run! size msecs b-out p-key t-key ta)))
