(ns olauzon.accumulator
  (:require [clojure.core.async :refer [>!!
                                        alt!!
                                        chan
                                        put!
                                        thread
                                        timeout]]))

(defn run!
  ([size msecs]
   (run! size msecs (chan) (chan) (chan)))
  ([size msecs e-in b-out c-in]
    "Buffer events accumulating from `e-in` for a maximum batch of `size` or
    `msecs` milliseconds. Batches are put in `b-out`."
    (let [s (dec size)]
      (thread
        (loop [batch []
               to    (timeout msecs)]
          (alt!!
            e-in ([e]
              (if (< (count batch) (dec size))
                (recur (conj batch e) to)
                (do
                  (>!! b-out (conj batch e))
                  (recur [] (timeout msecs)))))
            c-in ([_]
              (when-not (empty? batch) (>!! b-out batch)))
            to ([_]
              (if (empty? batch)
                (recur batch (timeout msecs))
                (do
                  (>!! b-out batch)
                  (recur [] (timeout msecs)))))))))
      {:e-in  e-in
       :b-out b-out
       :c-in  c-in
       :size  size
       :msecs msecs}))

(defn close!
  [acc]
  "Asynchronously close accumulator."
  (put! (:c-in acc) 0))
