(ns ring-gatekeeper.authenticators.auth0
  (:require [ring-gatekeeper.authenticators.core :as auth]
            [clj-http.client :as client]
            [clojure.data.json :as json]
            [ring-gatekeeper.cache.core :as cache]
            [ring-gatekeeper.cache.noop :as noop-cache]
            ))

(defonce _jwt_
  ; JWT require byte-streams, which has bugs which only allow it to be required
  ; once. This works around that by only requiring JWT once.
  (do
    (require '[jerks-whistling-tunes.core :as jwt]
             '[jerks-whistling-tunes.sign :refer [hs256]]
             '[jerks-whistling-tunes.utils :refer [decode-base-64]])))

(defn auth0-url [subdomain]
  (str "https://" subdomain ".auth0.com/"))

(defn auth0-token-info-url [auth0-url]
  (str auth0-url "userinfo"))

; Key must be unique per user per application
(defn build-cache-key [token]
  token)

(defn cache-user-info-response [cache cache-key info]
  (cache/set cache cache-key info))

(defn extract-claims [token client-secret client-id subdomain]
  (jwt/validate token
                (jwt/signature (hs256 client-secret))
                (jwt/aud client-id)
                (jwt/iss (auth0-url subdomain))
                jwt/exp))

(defn get-auth0-user-info-response [cache token auth0-url]
  (let [url (auth0-token-info-url auth0-url)
        response (client/get url
                             {:throw-exceptions false
                              :content-type :json
                              :headers {"Authorization" (str "Bearer " token)}
                              :cookie-policy :standard})
        {:keys [body status]} response]
    (if (= 200 status)
      (do
        (cache-user-info-response cache (build-cache-key token) body)
        body)
      nil)))

(defn get-cached-user-info-response [cache token]
  (cache/get cache (build-cache-key token)))

(defn extract-token-from-auth-header [auth-header]
  (let [[_ token] (re-matches #"^Bearer (.*)$" auth-header)]
    token))

(defn extract-token [request]
  (let [auth-header (get-in request [:headers "authorization"] "")]
    (or (extract-token-from-auth-header auth-header)
        (get-in request [:params "token"]))))

(def default-cache (noop-cache/new-cache))

(defrecord Auth0Authenticator [client-secret opts]
  auth/Authenticator

  (handle-request? [this request]
    ((:can-handle-request-fn opts) request))

  (authenticate [this request]
    (let [{:keys [client-id subdomain]} opts
          cache (get opts :cache default-cache)
          token (extract-token request)]
      (or (get-cached-user-info-response cache token)
          (get-auth0-user-info-response cache
                                        token
                                        (or (:auth0-url opts)
                                            (auth0-url subdomain)))))))

(defn new-authenticator [opts]
  (Auth0Authenticator. (decode-base-64 (or (:client-secret opts) "")) opts))
