; This Source Code Form is subject to the terms of the Mozilla Public
; License, v. 2.0. If a copy of the MPL was not distributed with this
; file, You can obtain one at https://mozilla.org/MPL/2.0/.

(ns noahtheduke.sinker
  (:require
   [noahtheduke.cond-plus :refer [cond+]]))

(set! *warn-on-reflection* true)

(defn parse-catch-clause
  [clause]
  (let [[_catch pred id & body] clause]
    (merge {:pred pred
            :id id
            :body body}
      (cond+
        [(keyword? pred)
         {:type :catch/type}]
        [(symbol? pred)
         (let [klass (resolve pred)]
           (if (class? klass)
             (if (= "clojure.lang.ExceptionInfo" (.getName ^Class klass))
               {:type :catch/raw}
               {:type :catch/class
                :pred klass})
             {:type :catch/fn}))]
        [(and (seq? pred) (symbol? (first pred)) (= "var" (name (first pred))))
         {:type :catch/fn}]
        [(nil? pred)
         (throw (ex-info "catch requires a class or predicate" {:clause clause}))]
        [:else
         (throw (ex-info "Incorrect catch clause" {:clause clause}))]))))

(defn parse-finally-clause
  [expr]
  {:type :finally/clause
   :body expr})

(defn parse-expr
  [expr]
  {:type :expr/clause
   :body expr})

(defn parse-clause
  [expr]
  (case (and (sequential? expr) (not (vector? expr)) (first expr))
    catch (parse-catch-clause expr)
    finally (parse-finally-clause expr)
    #_:else (parse-expr expr)))

(defn emit-exprs
  [exprs]
  (when (seq exprs)
    `(let [ret# (do ~@(map :body exprs))] ret#)))

(defn emit-catches
  [t ex-info? data catch-clauses]
  (for [{:keys [type pred id body]} catch-clauses]
    (case type
      :catch/type `[(and ~ex-info? (isa? ~pred (or (::type ~data) (:type ~data))))
                    (let [data# (or (ex-data ~t) {})
                          ~id (vary-meta data# assoc ::exception ~t)]
                      ~@body)]
      :catch/fn `[(and ~ex-info? (~pred ~data))
                  (let [data# (or (ex-data ~t) {})
                        ~id (vary-meta data# assoc ::exception ~t)]
                    ~@body)]
      :catch/raw `[~ex-info?
                   (let [~id ~t]
                     ~@body)]
      :catch/class `[(instance? ~pred ~t)
                     (let [~id ~t]
                       ~@body)])))

(defn emit-finallys
  [finally-clauses]
  (map :body finally-clauses))

(defn index-of
  [coll pred]
  (let [len (count coll)]
    (loop [i 0]
      (when-let [item (nth coll i nil)]
        (cond+
          [(pred item) i]
          [(= i len) nil]
          [:else (recur (inc i))])))))

(defn split-vec
  "Returns a vector of the given vec split at `idx`. If `idx` is `nil`, returns [vec nil]."
  [vec idx]
  [(if idx (subvec vec 0 idx) vec)
   (when idx (subvec vec idx (count vec)))])

(defn split-clauses
  [clauses]
  (let [idx (index-of clauses #(not= "expr" (namespace (:type %))))
        [exprs clauses] (split-vec clauses idx)
        idx (index-of clauses #(not= "catch" (namespace (:type %))))
        [catch-clauses finally-clauses] (split-vec clauses idx)]
    [exprs catch-clauses finally-clauses]))

(defmacro try+
  [& body]
  (when (seq body)
    (let [[exprs catch-clauses finally-clauses] (split-clauses (mapv parse-clause body))
          t (gensym "throwable_")
          ex-info? (gensym "ex-info?_")
          data (gensym "ex-data_")]
      `(try ~(emit-exprs exprs)
         ~@(when (seq catch-clauses)
             [`(catch Throwable ~t
                 (let [~ex-info? (instance? clojure.lang.ExceptionInfo ~t)
                       ~data (ex-data ~t)]
                   (cond+
                     ~@(emit-catches t ex-info? data catch-clauses)
                     [:else (throw ~t)])))])
         ~@(emit-finallys finally-clauses)))))
