(ns crimson.util
  (:require [clojure.string :as string]
            [clojure.walk :as walk]))

(defn parse-host-string
  [host-string]
  (let [tokens (string/split host-string #":")]
    (case (count tokens)
      1 {:host host-string}
      2 (let [[host ^String port] tokens]
          {:host host
           :port (Integer. port)})
      3 (let [[_ host ^String port] tokens]
          {:host (string/replace host "/" "")
           :port (Integer. port)})
      (throw (IllegalStateException.
              (str "Expected 'host', 'host:port' or 'scheme://host:port' but found "
                   host-string))))))

(defn sanitise-clojure-map-for-jedis
  "Converts the keywords to strings and applies the function `value-converter` on the values"
  [value-converter clojure-map]
  (reduce-kv
   (fn [m k v]
     (assoc m (name k) (value-converter v)))
   {} clojure-map))

(def sanitise-clojure-map-for-jedis-zadd (partial sanitise-clojure-map-for-jedis double))
(def sanitise-clojure-map-for-jedis-hset (partial sanitise-clojure-map-for-jedis str))

(defn java-hashmap->clojure-map
  "Converts a Java hashmap to a clojure map"
  [java-hashmap]
  (into {} java-hashmap))

(defn jedis-set-tuples->vector-tuples
  "Converts redis tuples to vector tuples"
  [redis-tuples]
  (mapv (fn [t] [(.getScore t) (.getElement t)])
        redis-tuples))
