(ns crimson.core
  (:refer-clojure :exclude [set get eval])
  (:require [crimson.pool :as pool]
            [crimson.util :as util])
  (:import [redis.clients.jedis Jedis JedisPool Transaction ScanParams]
           [redis.clients.jedis.exceptions JedisDataException]
           [redis.clients.jedis.params SetParams]))

(def iteration-start-cursor "0")

(defmacro with-conn
  "Evaluates body in a try-catch expression with the provided name
   bound to a connection resource from the pool and finally returns
   the connection back to the pool."
  [[var-sym redis-pool] & body]
  `(with-open [~(vary-meta var-sym assoc :tag "redis.clients.jedis.Jedis")
               (pool/connection ~redis-pool)]
     ~@body))

(defn set
  "Writes a value against a key to redis.

   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - k: a string key
    - v: a string value"
  ([conn-or-pool ^String k ^String v]
   (set conn-or-pool k v {}))

  ([conn-or-pool ^String k ^String v {:keys [ttl] :as options}]
   (if (instance? JedisPool conn-or-pool)
     (with-conn [conn conn-or-pool]
       (set conn k v options))
     (if (not (nil? ttl))
       (->> ttl
            (.ex (SetParams.))
            (.set conn-or-pool k v))
       (.set conn-or-pool k v)))))

(defn mset
  "Sets the given keys to their respective values.

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - key-value-pairs: A list of key-value pairs"
  [conn-or-pool key-value-pairs]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (mset conn key-value-pairs))
    (.mset conn-or-pool (into-array String key-value-pairs))))

(defn get
  "Reads a key from redis.

   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - k: a string key"
  [conn-or-pool ^String k]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (get conn k))
    (.get conn-or-pool k)))

(defn mget
  "Returns the values of all specified keys.
  For every key that does not hold a string value or does not exist, the special value nil is returned.

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - keys: A list of keys"
  [conn-or-pool keys]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (mget conn keys))
    (.mget conn-or-pool (into-array String keys))))

(defn scan
  "Returns a set of all keys that match the provided pattern https://redis.io/commands/scan
  Params:
   - conn-or-pool: An instance of `Jedis`/`JedisPool`
   - key-regex: a string key pattern"
  [conn-or-pool key-pattern]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (scan conn key-pattern))
    (let [scan-params (.match (ScanParams.) key-pattern)]
      (loop [cursor iteration-start-cursor
             result []]
        (let [scan-result (.scan conn-or-pool cursor scan-params)
              data        (.getResult scan-result)
              cursor      (.getCursor scan-result)]
          (if (.isCompleteIteration scan-result)
            (-> result (concat data) distinct vec)
            (recur cursor (-> result (concat data) vec))))))))

(defn delete
  "Deletes a key from redis.

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - k: a string key"
  [conn-or-pool ^String k]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (delete conn k))
    (.del conn-or-pool k)))

(defn incr
  "Increments the numeric value of a key.

  https://redis.io/commands/incr

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - k: a string key"
  [conn-or-pool ^String k]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.incr conn k))
    (.incr conn-or-pool k)))

(defn decr
  "Decrements the numeric value of a key. https://redis.io/commands/decr

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - k: a string key"
  [conn-or-pool ^String k]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.decr conn k))
    (.decr conn-or-pool k)))

(defn expire
  "Sets the TTL of a key. https://redis.io/commands/expire

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - k: a string key
    - seconds: the TTL in seconds"
  [conn-or-pool ^String k seconds]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.expire conn k seconds))
    (.expire conn-or-pool k seconds)))

(defn expire-at
  "Sets the TTL of a key to expire at a particular time. https://redis.io/commands/expireat

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - k: a string key
    - unix-timestamp: the Unix epoch representing the expiration date, as a Long"
  [conn-or-pool ^String k unix-timestamp]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.expireAt conn k unix-timestamp))
    (.expireAt conn-or-pool k unix-timestamp)))

(defn exists?
  "Check if a key exists. Returns true/false. https://redis.io/commands/exists

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - k: a string key"
  [conn-or-pool ^String k]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.exists conn k))
    (.exists conn-or-pool k)))

(defn eval
  "Evaluates a Lua script in Redis. https://redis.io/commands/eval

  Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`
    - script: the Lua script to be run
    - keys: a list of keys to be used as arguments to the script
    - args: a list of arguments (which are not keys) to be used by the script"
  [conn-or-pool ^String script keys args]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.eval conn script keys args))
    (.eval conn-or-pool script keys args)))

(defn flush-all!
  "Removes all keys from all databases in redis

   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`"
  [conn-or-pool]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (flush-all! conn))
    (.flushAll conn-or-pool)))


;; == Transactions ==


(defmacro with-multi
  "Evaluates body inside a redis `MULTI` transaction. The provided name
   is bound to a resource from the pool, using which a transaction
   is created. The connection will be returned back to the pool at
   the end of the transaction.

   Also, see `discard-transaction`.

   Refer https://redis.io/topics/transactions for details on how
   transactions work."
  [[var-sym ^JedisPool pool] & body]
  `(with-open [conn# (pool/connection ~pool)]
     (let [~(vary-meta var-sym assoc :tag "redis.clients.jedis.Transaction")
           (.multi conn#)]
       ~@body
       (try
         (.exec ~var-sym)
         (catch JedisDataException e# nil)))))

(defn discard-transaction
  "Discards the current transaction when called inside a `with-multi` block."
  [^Transaction transaction]
  (.discard transaction))

;; Sets Operation
(defn sadd
  "Add the specified members to the set stored at key

   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - k: a string key
    - v: a variadic string values"
  [conn-or-pool ^String key & members]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.sadd conn key (into-array String members)))
    (.sadd conn-or-pool key (into-array String members))))

(defn smembers
  "Returns all the members of the set value stored at key

   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - k: a string key"
  [conn-or-pool ^String key]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.smembers conn key))
    (.smembers conn-or-pool key)))

(defn srem
  "Remove the specified members from the set stored at key

   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - k: a string key
    - v: a variadic string values"
  [conn-or-pool ^String key & members]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.srem conn key (into-array String members)))
    (.srem conn-or-pool key (into-array String members))))

;; Sorted Sets Operation
(defn zadd
  "Writes a sorted set against a key to redis

   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - k: a string key
    - v: value as a clojure map of the following format `{^String :member ^Double score}`"
  [conn-or-pool ^String k v]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.zadd conn k (util/sanitise-clojure-map-for-jedis-zadd v)))
    (.zadd conn-or-pool k (util/sanitise-clojure-map-for-jedis-zadd v))))

(defn zrange
  "Return a range of members in a sorted set, by index.
   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - k: a string key
    - start: an integer index
    - end: an integer index. They can also be negative numbers indicating offsets from the end of the sorted set"
  [conn-or-pool ^String k start end]
  (into [] (if (instance? JedisPool conn-or-pool)
             (with-conn [conn conn-or-pool]
               (.zrange conn k start end))
             (.zrange conn-or-pool k start end))))

(defn zrange-with-score
  "Return a range of members in a sorted set, by index along with their scores
   Each element in the collection returned in the format: `[^Double score ^String member]`
   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - k: a string key
    - start: an integer index
    - end: an integer index. They can also be negative numbers indicating offsets from the end of the sorted set"
  [conn-or-pool ^String k start end]
  (-> (if (instance? JedisPool conn-or-pool)
        (with-conn [conn conn-or-pool]
          (.zrangeWithScores conn k start end))
        (.zrangeWithScores conn-or-pool k start end))
      (util/jedis-set-tuples->vector-tuples)))

;; Hashes Operation
(defn hset
  "Writes a clojure map against a key in redis
   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - k: a string key
    - value: a clojure map"
  [conn-or-pool ^String k value]
  (let [sanitized-value (util/sanitise-clojure-map-for-jedis-hset value)]
    (if (instance? JedisPool conn-or-pool)
      (with-conn [conn conn-or-pool]
        (.hset conn k sanitized-value))
      (.hset conn-or-pool k sanitized-value))))

(defn hget
  "Returns the value associated with field in the hash stored at key
   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - key: a string key
    - field: a string field"
  [conn-or-pool ^String key ^String field]
  (if (instance? JedisPool conn-or-pool)
    (with-conn [conn conn-or-pool]
      (.hget conn key field))
    (.hget conn-or-pool key field)))

(defn hdel
  "Removes the specified fields from the hash stored at key
   Params:
    - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
    - key: a string key
    - field: a variadic string fields"
  [conn-or-pool ^String key & fields]
  (let [array-fields (into-array String fields)]
    (if (instance? JedisPool conn-or-pool)
      (with-conn [conn conn-or-pool]
        (.hdel conn key array-fields))
      (.hdel conn-or-pool key array-fields))))

(defn hgetall
  "Returns all fields and values of the hash stored for a given key
  - conn-or-pool: An instance of `Jedis`/`JedisPool`/`JedisCluster`
  - k: a string key"
  [conn-or-pool ^String k]
  (-> (if (instance? JedisPool conn-or-pool)
        (with-conn [conn conn-or-pool]
          (.hgetAll conn k))
        (.hgetAll conn-or-pool k))
      (util/java-hashmap->clojure-map)))