# Crimson

A simple Redis client for Clojure built atop [Jedis](https://github.com/xetorthio/jedis).

See [rationale.md](doc/rationale.md) to understand the reason for Crimson's existence.

## Installation

Add Crimson to your `:dependencies`:

[//]: # (TODO: deploy under com.nilenso and update this)
`[com.nilenso/crimson "1.1.0"]`

## Usage

Require the following namespaces:

``` clojure
(ns your.ns
  (:require [crimson.core :as redis]
            [crimson.pool :as redis-pool]
            [crimson.cluster :as redis-cluster]))
```

## Examples

``` clojure
;; Initialise a redis connection pool
(def pool
  (redis-pool/init "localhost"
                   {:max-total                 10
                    :max-idle                  10
                    :min-idle                  50
                    :max-wait-millis           5000
                    :min-evictable-time-millis 300000}))

;; Do some operations
;; NOTE: Only string key-value operations are currently supported
(redis/set pool "foo" "bar")  ;; => "OK"
;; OR set key with expiry set to 60 seconds
(redis/set pool "foo" "bar" {:ttl 60})

(redis/get pool "foo")  ;; => "bar"
(redis/delete pool "foo")  ;; => 1

;; Close the pool when you're done
(redis-pool/close pool)

;; The simple approach above will work but it will fetch a new connection resource
;; from the pool for every operation. If you want to reuse a connection, use
;; `with-conn` instead. After executing the specified commands, the connection
;; is returned to the pool
(redis/with-conn [conn pool]
  (redis/set conn "name" "Bob")
  (redis/set conn "age" "22"))

;; If you're looking to run commands inside a redis transaction, use `with-multi`
(redis/with-multi [transaction pool]
  (redis/set conn "name" "Sam")
  (redis/delete conn "age"))  ;; => ["OK" 1]

;; If at any point, you wish to discard a transaction, call `discard-transaction`
;; on the connection inside a `with-multi` block
(redis/with-multi [conn pool]
  (redis/set conn "name" "Sam")
  (redis/delete conn "age")
  (redis/discard-transaction conn)) ;; => nil

;; ==============
;; Advanced Usage
;; ==============

;; Start a pooled connection to a redis cluster
(redis-cluster/init {:nodes              #{{:host "localhost" :port 6379}}
                     :pool-config        {:max-total                 100
                                          :max-idle                  100
                                          :min-idle                  50
                                          :max-wait-millis           5000
                                          :min-evictable-time-millis 300000}
                     :connection-timeout 2000
                     :socket-timeout     2000
                     :max-attempts       3})
```

## Contributing

Most of the code for this client was extracted from an internal project, and hence is tailored to its requirements.

Feel free to send PRs for any Redis feature that you may want to support.

### Development

- Clone the repository using `git clone https://github.com/nilenso/crimson.git`
- Setup Redis on your local machine
  - Install Redis using `brew install redis`
  - Start Redis server using `brew services start redis`

### Testing

Run the tests using:

``` shell
lein do clean, test
```

## License

Crimson is distributed under the Eclipse Public License version 2.0.
