(ns thrifty.service
  "Multimethods for transforming or inspecting Thrift Service protocols.

  When the protocol is generated for a service, the only methods it receives are part of the actual
  service. For interoperability, we need to translate that protocol into Java instances. We may also
  want to inspect Java-related classes and properties. These methods allow for that transformation
  while avoiding adding ad-hoc methods to the service protocols at compile-time."
  (:require [clojure.string :as str]
            [camel-snake-kebab.core :refer [->kebab-case]]))

(defn as-keyword
  "Returns namespaced keyword for an implementation of a service protocol.

    (def foo (reify MyService ...))
    (as-keyword foo) => :my.api/my-service"
  [impl]
  (let [fullname (-> impl (.getClass) (.getInterfaces) first (.getName))
        parts (str/split fullname #"\.")
        ns (str/replace (str/join "." (butlast parts)) "_" "-")
        n (->kebab-case (last parts))]
    (keyword ns n)))

(defn- dispatch [thing & rest]
  (as-keyword thing))

(defmulti handler
  "Return a Java handler, i.e., implementation of the actual Service interface."
  #'dispatch)

(defmulti processor
  "Return the service-side equivalent of `handler'; a TProcessor."
  #'dispatch)

(defmulti client
  "Return a client that transparently proxies calls to the actual service Client."
  #'dispatch)

(defmulti java-meta
  "Return an implementation of `thrift.generator.api/ServiceMeta'; limited public usefulness."
  #'dispatch)
