;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.reflector
  "Inspects a Java namespace and builds a representation using `thrifty.parser.schemas' records.

  The IDL parser already did the hard work of defining the Thrift language in terms of schemas
  so instead of reinventing that specifically for generating Clojure code, this module simply
  reuses the schemas. The logic only differs in where the information is coming from: Instaparse's
  IDL parse result or Thrift's generated Java namespace."
  (:require [clojure.set :refer [difference map-invert]]
            [thrifty.parser.schemas :as s])
  (:import com.google.common.base.Predicate
           org.apache.thrift.TFieldRequirementType
           [org.apache.thrift.meta_data ListMetaData MapMetaData EnumMetaData StructMetaData FieldMetaData]
           org.reflections.Reflections
           org.reflections.ReflectionUtils
           [org.reflections.scanners SubTypesScanner Scanner]
           java.lang.reflect.Modifier))

(def ^:private ttypes
  "Mapping of Thrift Type IDs to keywords.
   (see: `org.apache.thrift.protocol.TType`)"
  { 0x0 :stop   0x1 :void   0x2 :bool
    0x3 :byte   0x4 :double 0x6 :i16
    0x8 :i32    0xa :i64    0xb :string
    0xc :struct 0xd :map    0xe :set
   0xf :list })

(def jtypes
  "Mapping of Java types to built-in Thrift types"
  {java.lang.Byte "byte"
   java.lang.Integer "i32"
   java.lang.Long "i64"
   java.lang.Double "double"
   java.lang.Short "i16"
   java.lang.Boolean "bool"
   java.lang.String "string"})

(def jprims
  "Mapping of Java primitive names to `jtypes' key"
  {"int" java.lang.Integer
   "boolean" java.lang.Boolean
   "string" java.lang.String
   "byte" java.lang.Byte
   "float" java.lang.Double
   "long" java.lang.Long
   "short" java.lang.Short
   "double" java.lang.Double})

(def jarrays
  "Set of all java primitive array types"
  (set (map #(Class/forName (str "[" %)) '(Z B C D F I J S))))

(def ^:private predicates
  {:public (ReflectionUtils/withModifier Modifier/PUBLIC)})

(defn pred [& preds]
  (into-array Predicate (map #(if (keyword? %) (% predicates) %) preds)))

(defn process-namespace [ns]
  (let [ref (Reflections. ns (into-array Scanner [(SubTypesScanner. false)]))
        classes (set (remove #(.getDeclaringClass %) (.getSubTypesOf ref java.lang.Object)))
        enums (set (.getSubTypesOf ref org.apache.thrift.TEnum))
        tbase (set (remove #(.getDeclaringClass %) (.getSubTypesOf ref org.apache.thrift.TBase)))
        exceptions (set (filter #(.isAssignableFrom org.apache.thrift.TException %) tbase))
        structs (difference tbase exceptions)
        services (filter (fn [i] (some #(.endsWith (.getName %) "$Iface") (.getDeclaredClasses i)))
                         (difference classes tbase))]
    {:structs structs
     :enums enums
     :exceptions exceptions
     :services services}))

(defn enum->map [^java.lang.Class e]
  (into {} (map (fn [i] [(keyword (clojure.string/lower-case (.getName i)))
                         (symbol (str (.getName e) "/" (.name (.get i nil))))])
                (ReflectionUtils/getAllFields e (pred :public)))))

(defn vmd->schema
  ([vm type-locator] (vmd->schema vm type-locator nil))
  ([vm type-locator ignore-typedef?]
   (let [type (.-type vm)
         type-name (when-let [kw (get ttypes type)] (name kw))
         tdn (.getTypedefName vm)
         m (cond
             (instance? StructMetaData vm)
             {:name :struct :value (.getSimpleName (.-structClass vm)) :class (.-structClass vm)
              :ns (type-locator :struct (.getSimpleName (.-structClass vm)))}

             (.isStruct vm)
             {:name :struct
              :value tdn
              :ns (type-locator :struct tdn)
              :class (ReflectionUtils/forName (str (type-locator :class tdn) "." tdn) nil)}

             (and (.isTypedef vm) (not ignore-typedef?))
             (s/as-type {:name tdn :type (vmd->schema vm type-locator true)
                         :ns (type-locator :typedef tdn)})

             (instance? ListMetaData vm)
             {:name :container :type :list :value [(vmd->schema (.-elemMetaData vm) type-locator)]}

             (instance? MapMetaData vm)
             {:name :container :type :map :value [(vmd->schema (.-keyMetaData vm) type-locator)
                                                  (vmd->schema (.-valueMetaData vm) type-locator)]}

             (instance? EnumMetaData vm)
             {:name :struct :value (.getSimpleName (.-enumClass vm)) :class (.-enumClass vm)}

             (.isBinary vm)
             {:name :builtin :value "binary" :class (class (byte-array 0))}

             :default
             {:name :builtin :value type-name :class (get (map-invert jtypes) type-name)})]
     (if-not (instance? thrifty.parser.schemas.TType m) (s/field-type m) m))))

(defn struct-field->schema [[key val] s type-locator]
   (let [ns (-> s (.getPackage) (.getName))]
     (s/field
      {:name (.getFieldName key)
       :id (.getThriftFieldId key)
       :required? (not (= TFieldRequirementType/OPTIONAL (.requirementType val)))
       :type (vmd->schema (.-valueMetaData val) type-locator)})))

(defn struct->schema [s type-locator]
   (let [mdm (-> (ReflectionUtils/getAllFields s (pred (ReflectionUtils/withName "metaDataMap")))
                 first
                 (.get nil))
         n (.getSimpleName s)]
     (s/as-struct {:name n
                   :ns (type-locator :struct n)
                   :class s
                   :fields (map #(struct-field->schema % s type-locator) mdm)})))

(defn function->schema [f type-locator]
  (let [mdm-pred (pred (ReflectionUtils/withName "metaDataMap"))
        success-pred (fn [[key val]] (= "success" (.-fieldName val)))
        classes (into {} (map (fn [i] [(.getSimpleName i) i])
                              (-> f (.getDeclaringClass) (.getDeclaringClass) (.getDeclaredClasses))))
        args-cls (get classes (str (.getName f) "_args"))
        res-cls (get classes (str (.getName f) "_result"))
        args-mdm (FieldMetaData/getStructMetaDataMap args-cls)
        res-mdm (FieldMetaData/getStructMetaDataMap res-cls)
        returns (first (filter success-pred res-mdm))]
    (s/as-fn { :name (.getName f)
              :returns (if returns (vmd->schema (.-valueMetaData (val returns)) type-locator) "void")
              :oneway? false ;; TODO: Set oneway? to actual value
              :args (for [[key val] args-mdm]
                      (s/field {:name (.getFieldName key)
                                :id (.getThriftFieldId key)
                                :type (vmd->schema (.-valueMetaData val) type-locator)}))
              ; TODO: Do something with `throws' maybe?
              :throws []})))

(defn service->schema [type-locator s]
   (let [iface (ReflectionUtils/forName (str (.getName s) "$Iface") nil)]
     (s/as-svc {:name (.getSimpleName s)
                :iface iface
                :methods (map #(function->schema % type-locator)
                              (ReflectionUtils/getAllMethods iface (pred :public)))})))
