;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.parser.schemas
  (:require [clojure.string :refer [join]]
            [schema.core :as s :refer [Bool cond-pre enum eq Int maybe Str]]))

(declare cft)

;; Fields

(s/defrecord ConstValue
    [type :- (enum :number :list :map :string :struct)
     value :- (cond-pre Str s/Num [] {})])

(s/defrecord BuiltinFieldType
    [value :- (enum "string" "bool" "byte" "i8" "i16" "i32" "i64"
                    "double" "string" "binary" "slist")
     class :- (maybe java.lang.Class)]
  java.lang.Object
  (toString [f] (:value f)))

(s/defrecord StructFieldType
    [value :- Str
     class :- (maybe java.lang.Class)
     ns :- (maybe Str)]
  java.lang.Object
  (toString [f]
    (:value f)))

(s/defrecord TType
    [name :- Str
     comment :- (maybe Str)
     ns :- (maybe Str)
     type :- (cond-pre BuiltinFieldType StructFieldType (s/recursive #'cft))])

(s/defn as-type :- TType [m]
  (map->TType m))

(def any-field-type (cond-pre TType BuiltinFieldType StructFieldType (s/recursive #'cft)))

(s/defrecord ContainerFieldType
    [type :- (enum :list :map :set)
     value :- [any-field-type]]
  java.lang.Object
  (toString [f]
    (str (name (:type f))
         "<"
         (if (= :map (:type f)) (join "," (map str (:value f))) (str (first (:value f))))
         ">")))

(s/defrecord EnumField
    [comment :- (maybe Str)
     name :- Str
     value :- (maybe Int)])

(def ^:private cft ContainerFieldType)

(s/defn field-type :- any-field-type
  [m]
  (if-not (instance? TType m)
    (let [name (:name m)
        m (dissoc m :name)]
    (condp = name
      :container (map->ContainerFieldType m)
      :builtin (map->BuiltinFieldType m)
      nil m  ;; Has no name; already parsed
      (map->StructFieldType m)))
    m))

(s/defrecord Field
    [id :- (maybe Int)
     comment :- (maybe Str)
     name :- Str
     value :- (maybe ConstValue)
     required? :- (maybe Bool)
     type :- any-field-type])

(def any-field (cond-pre Field EnumField))

(s/defn field :- any-field [m]
  (if-let [type (:type m)]
    (map->Field (assoc m :type (field-type type)))
    (map->EnumField m)))

;; Const

(s/defrecord TConst
    [name :- Str
     type :- any-field-type
     comment :- (maybe Str)
     value :- ConstValue])

(s/defn as-const :- TConst [m]
  (map->TConst m))

;; Struct

(s/defrecord TStruct
    [name :- Str
     ns :- (maybe Str)
     comment :- (maybe Str)
     class :- (maybe java.lang.Class)
     fields :- [any-field]])

(s/defn as-struct :- TStruct [m]
  (TStruct. (:name m) (:ns m) (:comment m) (:class m) (sort-by :id (:fields m))))

;; Enum

(s/defrecord TEnum
    [name :- Str
     comment :- (maybe Str)
     fields :- [EnumField]])

(s/defn as-enum :- TEnum [m]
  (map->TEnum m))

;; Exception

(s/defrecord TException
    [name :- Str
     comment :- (maybe Str)
     fields :- [any-field]])

(s/defn as-exc :- TException [m]
  (TException. (:name m) (:comment m) (sort-by :id (:fields m))))

;; Function

(s/defrecord TFunction
    [name :- Str
     comment :- (maybe Str)
     returns :- (s/if string? (eq "void") any-field-type)
     oneway? :- Bool
     args :- [any-field]
     throws :- [any-field]])

(s/defn as-fn :- TFunction [m]
  (map->TFunction m))

;; Service

(s/defrecord TService
    [name :- Str
     iface :- (maybe java.lang.Class)
     comment :- (maybe Str)
     methods :- [TFunction]])

(s/defn as-svc :- TService [m]
  (map->TService m))
