;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.parser.field
  (:require [clojure.string :as str]
            [puppetlabs.kitchensink.core :refer [cond-let parse-number]]
            [thrifty.parser.schemas :as s]
            [thrifty.parser.util
             :refer
             [find-identifier find-next-tag find-next-tags find-tag]])
  (:import thrifty.parser.schemas.ConstValue))

(defmulti handle-field-type
  "Post-process a `[:field-type ...]' form."
  (fn [form]
    (let [type-name (first (second form))]
      (cond (= :container-type type-name) :container
            (= :base-type type-name)      :builtin
            :else                         :struct))))

(defmethod handle-field-type :builtin [field] (s/field-type {:name :builtin :value (second (second field))}))
(defmethod handle-field-type :struct [field] (s/field-type {:name :struct :value (second (second field))}))
(defmethod handle-field-type :container [field]
  (let [sub-field (second (second field))
        field-type (keyword (str/replace (name (first sub-field)) "-type" ""))]
    (s/field-type
     {:name :container
      :type field-type
      :value (if (= :map field-type)
               (apply list (map handle-field-type (rest sub-field)))
               [(handle-field-type (second sub-field))])})))

(defmulti handle-const-value #(first (second %)))

(defmethod handle-const-value :const-list [field]
  (ConstValue. :list (->> field
                          (find-next-tag :const-list)
                          (find-next-tags :const-value)
                          (map #'handle-const-value)
                          (map :value)
                          (apply list))))

(defmethod handle-const-value :const-map [field]
  (ConstValue. :map (->> field
                         (find-next-tag :const-map)
                         (find-next-tags :const-value)
                         (map #'handle-const-value)
                         (map :value)
                         (apply hash-map))))

(defmethod handle-const-value :int-constant [field]
  (ConstValue. :number (parse-number (second (find-tag :digit field)))))

(defmethod handle-const-value :double-constant [field]
  (ConstValue. :number (parse-number (second (find-tag :digit field)))))

(defmethod handle-const-value :identifier [field]
  (ConstValue. :struct (second (find-next-tag :identifier field))))

(defmethod handle-const-value :literal [field]
  (ConstValue. :string (second (find-next-tag :literal field))))

(defmethod handle-const-value :default [field] nil)

(defn handle-field
  "Post-process a `[:field ...]' form"
  [f]
  (let [ftype (find-next-tag :field-type f)]
    (s/field
     {:comment (when-let [c (find-next-tag :comment f)] (second c))
      :id (when-let [id (find-next-tag :field-id f)] (Integer/parseInt (second (find-tag :digit id))))
      :required? (when-let [[_ r] (find-next-tag :field-req f)] (if (= "optional" r) false true))
      :value (when-let [v (find-next-tag :const-value f)] (handle-const-value v))
      :type (handle-field-type ftype)
      :name (second (find-next-tag :identifier f))})))

(defn handle-enum-field [f]
  (let [val (second (rest f))]
    (s/field
     {:name (find-identifier f)
      :comment (when-let [c (find-next-tag :comment f)] (second c))
      :value (when-let [v (find-tag :digit f)]
               (Integer/parseInt (second v)))})))
