;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.parser.definition
  "Transformers for top-level containers; a struct, enum, service, etc.

  Most definitions are just bags of fields so the logic is pretty
  similar for all of them."
  (:require [thrifty.parser.field
             :refer
             [handle-const-value handle-enum-field handle-field handle-field-type]]
            [thrifty.parser.schemas
             :refer
             [as-const as-enum as-exc as-fn as-struct as-svc as-type]]
            [thrifty.parser.util
             :refer
             [find-identifier find-next-tag find-next-tags find-tag find-tags]]))

(declare handle-function)

(defmulti handle-definition
  "Post-process a `[:definition ...]' form.

  Method is dispatched on the wrapped definition, e.g., `:struct' in [:definition [:struct ...]]."
  (fn [form] (first (second form))))

(defmethod handle-definition :struct [[_ form]]
  (as-struct
   {:name (find-identifier form)
    :comment (when-let [c (find-next-tag :comment form)] (second c))
    :fields (map #'handle-field (find-next-tags :field form))}))

(defmethod handle-definition :const [[_ form]]
  (as-const
   {:type (handle-field-type (find-tag :field-type form))
    :comment (when-let [c (find-next-tag :comment form)] (second c))
    :value (handle-const-value (find-tag :const-value form))
    :name (find-identifier form)}))

(defmethod handle-definition :typedef [[_ form]]
  (as-type
   {:type (handle-field-type (find-tag :definition-type form))
    :comment (when-let [c (find-next-tag :comment form)] (second c))
    :name (find-identifier form)}))

(defmethod handle-definition :enum [[_ form]]
  (as-enum
   {:name (find-identifier form)
    :comment (when-let [c (find-next-tag :comment form)] (second c))
    :fields (->> (find-tags :enum-field form)
                 (map handle-enum-field))}))

(defmethod handle-definition :exception [[_ form]]
  (as-exc
   {:name (find-identifier form)
    :comment (when-let [c (find-next-tag :comment form)] (second c))
    :fields (map handle-field (find-next-tags :field form))}))

(defmethod handle-definition :service [[_ form]]
  (as-svc
   {:name (find-identifier form)
    :comment (when-let [c (find-next-tag :comment form)] (second c))
    :methods (map #'handle-function (find-next-tags :function form))}))

(defn handle-function [form]
  (as-fn
   {:name (find-identifier form)
    :returns (let [t (find-next-tag :function-type form)]
               (if-let [f (find-next-tag :field-type t)]
                 (handle-field-type f)
                 (second t)))
    :oneway? (contains? (set form) "oneway")
    :comment (when-let [c (find-next-tag :comment form)] (second c))
    :args (map handle-field (find-next-tags :inline-field form))
    :throws (when-let [throws (find-next-tag :throws form)]
              (map handle-field (find-tags :inline-field throws)))}))

(defmethod handle-definition :default [v]  v)
