;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.parser.core
  "Main parser API and transformation pipeline definition.

  The process for parsing a Thrift specification is:

    1. Run through `instaparse', per thrift.bnf
    2. Process the resulting hiccup forms, turning them into
       structures that are easier to use.
    3. Shove result of (2) into `prismatic/schema' definitions.

  For logic related to processing individual IDL pieces (definitions,
  fields, and etc.) see siblings modules."
  (:require [clojure.string :as s]
            [com.rpl.specter :refer [transform walker]]
            [instaparse.core :as insta]
            [thrifty.parser.definition :refer [handle-definition]]
            [thrifty.parser.util
             :refer
             [find-next-tag find-next-tags match-hiccup]]))

(declare clean-sl-comments clean-ml-comments ->map)

(def ^:private parser
  "Shared instaparse parser"
  (insta/parser (clojure.java.io/resource "thrift.bnf")))

(def ^:private default-transformers
  "Pairs of `(walker-fn . transform-fn)'; used via `specter'"
  [[(match-hiccup :ml-comment) #'clean-ml-comments]
   [(match-hiccup :sl-comment) #'clean-sl-comments]
   [(match-hiccup :definition) #'handle-definition]])

(defn parse
  "Primary entrypoint into the IDL parser.
  With :raw, don't do any transformation of the instaparse result."
  [idl & {:keys [raw transformers] :or [false]}]
  (let [tree (cond (string? idl) (parser idl)
                   :else (parser (slurp idl)))
        pipeline (apply comp (map (fn [[m x]] (partial transform (walker m) x))
                                  (or transformers default-transformers)))]
    (if raw tree
        (let [result (pipeline tree)]
          (if-not transformers (->map result) result)))))

(defn- ->map [result]
  (merge
   (when-let [c (find-next-tag :comment result)]
     (let [lines (s/split (second c) #"\n")
           name (s/trim (first lines))]
       {:name (s/lower-case name)
        :display-name name
        :description (s/trim (s/join "\n" (rest lines)))}))
   {:definitions (remove vector? (rest result))}))

(defn clean-sl-comments
  "Reduce `comment-line' forms in :sl-comment"
  [form]
  (-> form
      second
      second
      (s/replace #"\s*[\*]+(/|$)" "")
      s/trim))

(defn clean-ml-comments
  "Strip the sugar from block comments."
  [val]
  (let [lines (s/split-lines (reduce #(s/join "\n" [%1 %2]) (map second (rest val))))]
    (s/trim (s/join "\n" (for [line lines]
                           (-> line
                               s/trim
                               (s/replace #"/[\*]+\s*" "")
                               (s/replace #"\s*[\*]+(/|$)" "")
                               (s/replace #"/\*\s*" "")
                               (s/replace #"^\s*\*\s+" "")))))))
