;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.generator.docs
  "Generate a Sphinx <http://www.sphinx-doc.org> document describing a Thrift API.

  The resulting file can be added to any Sphinx project and will compile to API documentation that
  looks similar to that provided by Sphinx's `autodoc' extension. To build the resulting the file,
  you'll need the thriftdomain <https://bitbucket.org/bio-plus/sphinx-thriftdomain> extension
  installed and configured in your Sphinx_ project.

  This project's source code includes a file `bin/thrift-doc' that creates an actual file whereas
  the `generate' function here returns the raw text that can be `spit' anywhere."
  (:require [clojure.string :refer [join split trim]]
            [selmer.filters :refer [add-filter!]]
            [selmer.parser :refer [render-file]]
            [selmer.util :refer [without-escaping]]
            [clojure.string :as s]))

(declare ->context)

(def ^:private cls->key
  {:TConst :constants
   :TEnum :enums
   :TType :typedefs
   :TStruct :structs
   :TService :services
   :TException :exceptions})

(defn generate
  "Returns text of an rST file documenting result of `thrifty.parser.core/parse'"
  [result & {:keys [template]}]
  (without-escaping
   (s/replace (render-file (or template "templates/thrifty/index.rst") (->context result))
              #"\n{3,}"
              "\n\n")))

(defn- ->context [parsed]
  (let [result (merge
           {:api (select-keys parsed [:name :display-name :description])}
           (group-by (fn [i] ((keyword (.getSimpleName (.getClass i))) cls->key)) (:definitions parsed)))]
    result))

(defn- fmt-directive-body
  ([s] (fmt-directive-body s "   "))
  ([s indent]
   (let [[head & tail] (split s #"\n")]
     (join "\n" (concat [head] (map #(str indent (trim %)) tail))))))

(defn- one-line [s]
  (if s (.replace (trim s) "\n" " ")))

(defn- signature [meth]
  (str (:name meth) "(" (join ", " (map :name (:args meth))) ")"
       (if-not (:oneway? meth) (str " -> " (:returns meth)))))

(add-filter! :dbody #'fmt-directive-body)
(add-filter! :oneline #'one-line)
(add-filter! :signature #'signature)
