;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.generator.api
  "Generate a Clojure Thrift API from a Java one.

  Call the `generate' macro with a Java namespace found on the classpath. The namespace in which
  you've called `generate' will contain the following:

    1. For each Enum, a map `(kebab-case TheEnum)' as `(kebab-case FIELD) -> FIELD'
    2. For each Struct, a `prismatic/schema' record representing the fields and the appropriate
       schema for each. The record is validated prior to conversion to the original Java object.
       It implements `JavaBoundary' to convert between Java-land and Clojure-land.
    3. For each Service, a protocol to implement for servers. Also, implementations of
       `thrifty.service' multi-methods dispatched as `(keyword ns (kebab-case Service))'.

  Usage:

    (generate \"acme.api\" {})  ;; Generate the API

    (def acme-service (reify AcmeService ...))  ;; Use pure Clojure to implement the protocol
    (def server (thrifty.connect/mk-server :threadpool acme-service 9000 \"localhost\" {:ssl true}))
    (thrifty.connect/serve! server)  ;; Run server in another thread
    (thrifty.connect/connect! acem-service 9000 \"localhost\" {:ssl true})  ;; Connect a client


  See documentation on the clj-* functions for more details on the generated code."
  (:require [camel-snake-kebab.core
             :refer
             [->camelCaseString ->kebab-case ->kebab-case-symbol]]
            [clojure.string :as st]
            [schema.core :as s]
            schema.macros
            [schema.spec.core :as spec]
            [schema.spec.variant :as variant]
            [schema.utils :refer [error-val error? format*]]
            [thrifty.coerce :as coerce :refer [java->clj]]
            [thrifty.reflector
             :refer
             [enum->map
              jarrays
              jprims
              process-namespace
              service->schema
              struct->schema]]
            [thrifty.service :as svc]
            [clojure.tools.logging :as log])
  (:import [clojure.lang Keyword Reflector]
           schema.core.Schema
           [thrifty.parser.schemas BuiltinFieldType ContainerFieldType StructFieldType TType]))

(declare varspec xplain builtin-type? primitive-array? pass-through? cap-first
         setter-method io maybe-error struct-schema? typedef-schema? ttype-val typedef? maybe)

(def ^:private bft->ttype
  {"string" s/Str
   "bool" s/Bool
   "byte" java.lang.Byte
   "i8" s/Int
   "i16" s/Int
   "i32" s/Int
   "i64" s/Num
   "double" s/Num
   "binary" (s/cond-pre (class (byte-array 0)) java.nio.ByteBuffer)})

(defprotocol ApiField
  "Convert `thrifty.parser.schemas' records."
  (field-schema [this ns] "Returns the appropriate prismatic/schema for a struct field type"))

(defprotocol JavaBoundary
  "Manage the boundary between Java and Clojure. Implementation emitted by macro."
  (to-java [this])
  (from-java [this obj]))

(defprotocol ServiceMeta
  "Mostly-internal methods for Java-related metadata about a service."
  (arg-class [this ^Keyword method] "Returns the symbol pointing to a java Class for a method's arguments")
  (method-args [this ^Keyword method] "Returns ordered vector of arguments to a method (as schema/Field instances)")
  (method-names [this] "Returns a set of method names as keywords "))

(extend-protocol s/Schema
  BuiltinFieldType
  (spec [this]
    (let [c (get bft->ttype (:value this))]
      (if (satisfies? s/Schema c) (s/spec c) (varspec c))))
  (explain [this]
    (str "a-" (xplain (get bft->ttype (:value this)))))

  StructFieldType
  (spec [this]
    (let [cls (:class this)]
      (if-not (.isAssignableFrom org.apache.thrift.TEnum cls)
        (s/spec (s/recursive (find-var (symbol (:ns this) (str "fwd-" (.getSimpleName cls))))))
        (varspec cls))))
  (explain [this] (str "a-" (xplain (:class this))))

  ContainerFieldType
  (spec [this]
    (s/spec
     (condp = (:type this)
       :list [(-> this :value first)]
       :set #{(-> this :value first)}
       :map {(-> this :value first)
             (-> this :value second)})))
  (explain [this]
    (let [t (name (:type this))]
      (if (= "map" t)
        (list 'a t 'of (xplain (first (:value this))) '-> (xplain (second (:value this))))
        (list 'a t 'of (xplain (first (:value this)))))))

  TType
  (spec [this]
    (s/spec (:type this)))
  (explain [this] (s/explain (:type this))))

(defn coercion-fn
  "Create a schema coercion function for a field or struct."
  ([^Schema rec] (coercion-fn rec coerce/noop coerce/noop))
  ([^Schema rec converter] (coercion-fn rec converter coerce/noop))
  ([^Schema rec converter callback]
   (spec/run-checker
    (fn [sch params]
      (let [walk (spec/checker (s/spec sch) params)]
        (fn [x]
          (loop [res (walk x)
                 retrying? false]
            (if (and (error? res) (not retrying?))
              (cond
                (struct-schema? sch)
                (recur (walk ((coerce/mapper sch) (converter sch x))) true)

                (typedef-schema? sch)
                (recur (walk (converter sch x)) true)

                :default (recur res true))

              (let [cnv #(if-not retrying? (converter sch %) %)]
                (cond
                  (error? res) res

                  (struct-schema? sch)
                  (callback sch (cnv res))

                  (typedef-schema? sch)
                  (let [r (cnv res)] (or (callback sch r) r))

                  :default res)))))))
    true
    rec)))

(defn ->java
  "Given a Clojure record and optional target Schema, coerce a Java object.
  If `schema' isn't present, the record itself should implement `Schema'.

  Args:
    - is-domain? The record is already in \"domain mode\"; the equivalent of calling `coerce/to-domain'.

  Java primitives and strings are passed through."
  ([rec] (->java rec true (class rec)))
  ([rec is-domain?] (->java rec is-domain? (class rec)))
  ([rec is-domain? schema]
   (if-not (builtin-type? rec)
     (let [converter (if is-domain? coerce/to-schema coerce/noop)
           callback (fn [_ res]
                      (when (satisfies? coerce/ApiStruct res)
                        (let [o (Reflector/invokeConstructor (coerce/java-class res) (io))]
                          (doseq [[field v] res]
                            (if-not (nil? v)
                              (Reflector/invokeInstanceMethod o (setter-method field) (io v))))
                          o)))]
       (maybe-error rec ((coercion-fn schema converter callback) rec) schema))
     rec)))

(defn ->clj
  "Given a target Schema and Java object, coerce a Clojure record/seq satisfying the schema."
  [schema obj]
  (let [converter coerce/to-domain]
    (maybe-error obj ((coercion-fn schema java->clj converter) (java->clj schema obj)) schema)))

(defn ->domain [schema]
  (coercion-fn schema coerce/noop coerce/to-domain))

(defn ->schema [schema]
  (coercion-fn schema coerce/to-schema))

(defn- clj-structs
  "Generate a `prismatic/schema' record for a Java 'struct' (class)

  Struct fields are convertered to kebab-case; otherwise the record functions just like any
  schema-emitted record. Additionally, you may wish to extend it using the `coerce' methods
  which will be used to convert to/from preferred domain-specific transformations. For instance:

    (defmethod coerce/eto-domain :my.ns/MyStruct
     [_ o] (assoc o :timestamp (java.sql.Timestamp. (:timestamp o))))
    (defmethod coerce/to-schema :my.ns/MyStruct
     [_ o] (assoc o :timestamp (.getTime (:timestamp o))))

  Now any `MyStruct' record you receive has a `:timestamp' key that is a SQL timestamp; the field
  will be converted back to a number before being set on the Java class and serialized by Thrift."
  [structs type-locator]
  (for [struct (map #(struct->schema % type-locator) structs)]
    (let [sym (symbol (:name struct))
          fwd (symbol (str "fwd-" (:name struct)))
          fields (map (comp ->kebab-case :name) (:fields struct))]
      `(do
         ;;(ns-unmap *ns* '~sym)
         ;;(ns-unmap *ns* '~fwd)
         (declare ~fwd)
         (s/defrecord ~sym
             ~(vec (mapcat (fn [f]
                             `(~(symbol (-> f :name ->kebab-case)) ~':-
                               ~(maybe f))) (:fields struct)))
           JavaBoundary
           (to-java [~'s] (->java ~'s))
           (from-java [~'s ~'obj] (->clj ~'s ~'obj))

           coerce/ApiStruct
           (coerce/struct-schema [~'this] ~struct)
           (coerce/java-class [~'this] (:class (coerce/struct-schema ~'this))))

         (def ^:private ~fwd ~sym)))))

(defn- clj-enums
  "Generate map-style enums for Java enums.

  For enum `MyEnum { FOO, BAR_BAZ }' a map `my-enum {:foo FOO :bar-baz BAR_BAZ} will be generated.'"
  [enums]
  (for [enum enums]
    (let [sym# (->kebab-case-symbol (.getSimpleName enum))]
      `(do
         ;;(ns-unmap *ns* '~sym#)
         (defonce ~sym# ~(enum->map enum))))))

(defn- cls-or-schema [ft ns]
  (if-let [cls (:class ft)]
    (let [tfull (.getName cls)
          tshrt (.getSimpleName cls)]
      (symbol (if (.startsWith tfull ns) tshrt tfull)))))

(defn- clj-methods
  "Generate a protocol method.

  Methods are converted to kebab-case and expect generated schemas over Java types."
  [service ns]
  (let [arg->param (fn [i] `[~(symbol (:name i))])]
    (for [meth (:methods service)]
      `(~(->kebab-case-symbol (:name meth))
        ~(vec (concat ['this] (vec (reduce concat (map arg->param (:args meth))))))))))

(defn- caller [ns client? c]
  (let [name (symbol (:name c))
        fn-sym (if client? (->kebab-case-symbol name) name)
        cfn-sym (if client? (symbol (str "." name)) (->kebab-case fn-sym))
        arg-fn (if client? ->java ->clj)
        ret-fn (if client? ->clj ->java)
        args (map (fn [i] [(:type i) (symbol (:name i))]) (:args c))
        arg-names (map (comp symbol second) args)
        call-args `(~cfn-sym ~'this ~@(for [[s o] args] (if client? `(~arg-fn ~o true ~s) `(~arg-fn ~s ~o))))]
    `(~fn-sym [~'_ ~@arg-names]
      (log/debugf "Called %s method `%s' with args: %s"
                  ~(if client? "client" "server") ~(str fn-sym) ~(vec arg-names))
      ~(if (or (= "void" (:returns c)) (= "void" (-> c :returns :value)))
         call-args
         `(let [retval# ~(if client?
                          `(~ret-fn ~(:returns c) ~call-args)
                          `(~ret-fn ~call-args true ~(:returns c)))]
            (log/debugf "Returning from %s method `%s': %s"
                        ~(if client? "client" "server") ~(str fn-sym) retval#)
            retval#)))))

(defn- clj-services
  "Generate a protocol from a Thrift service, with the following specification:

  1. The service name itself will become the protocol
  2. Method names are converted to kebab-case.
  3. Method arguments referring to structs (individually or in collections) receive their schema equivalent.
  4. Methods returning structs (individually or in collections) return their schema equivalent.

  Basically, implementing your protocol should not involve any Java types.

  Additionally, the following top-level functions are included for convenience:

  1. `<service>-handler' converts a Protocol implementation to a Java handler instance.
  2. `<service>-client' connects clients to the service; takes 1-2 arguments of type TProtocol."
  [services type-locator]
  (for [svc (map (partial #'service->schema type-locator) services)]
    (let [ns (-> svc :iface (.getPackage) (.getName))
          sym (symbol (:name svc))
          csym (symbol (str (:name svc) "Client"))
          client (symbol (str ns "." (:name svc) "$Client"))
          processor (symbol (str ns "." (:name svc) "$Processor"))
          arg (->kebab-case-symbol sym)
          dispatch (keyword (str *ns*) (str arg))
          iface (:iface svc)]
      `(do
         ;;(ns-unmap *ns* '~sym)

         (defprotocol ~sym
           ~(str "Protocol for implementing Thrift service `" (:name svc) "'")
           ~@(clj-methods svc ns))

         (defmethod svc/java-meta ~dispatch [~'this]
           (reify ServiceMeta
             (method-names [this#] ~(set (map (comp keyword :name) (:methods svc))))
             (method-args [this# kw#]
               (condp = kw#
                 ~@(reduce concat (for [meth (:methods svc)]
                                    [(-> meth :name keyword) (vec (:args meth))]))))
             (arg-class [this# kw#]
               (condp = kw#
                 ~@(reduce concat (for [n (map (comp keyword :name) (:methods svc))]
                                    [n (symbol (str ns "." sym "$" (name n) "_args"))]))))))

         (defmethod svc/handler ~dispatch [~'this]
           (reify ~(symbol (.getName iface)) ~@(map (partial caller ns false) (:methods svc))))

         (defmethod svc/client ~dispatch [~'this ~'proto]
           (let [~'this (new ~client ~'proto)]
             (reify
               ~sym
               ~@(map (partial caller ns true) (:methods svc))

               java.io.Closeable
               (close [~'_] (-> ~'this (.getInputProtocol) (.getTransport) (.close))))))

         (defmethod svc/processor ~(keyword (str *ns*) (str arg)) [~'this]
           (new ~processor (svc/handler ~'this)))
         nil
         ))))

(defn- clj-requires [type-map]
  (for [[ns types] type-map]
    (let [syms (map #(->> % name (str "fwd-") symbol) types)]
      `(do
         ;; (doall (map (partial ns-unmap *ns*) '~syms))
         (require '~(vec (list (symbol (name ns)) :refer (vec syms))))))))

(defmacro generate
  "Generate a Clojure/Thrift API from an existing Java namespace.

  The `type-map' argument must be used if the namespace references
  structs in different namespaces. Since this macro doesn't know where
  the java namespace \"com.foo.my_other_api\" may end up in Clojure,
  you'll need to tell it, as a map of Clojure namespace to types:

    {:foo.my-other-api #{:Type1 :Type2}}

  (All vars produced by this macro are interned at the calling
  namespace.)"
  [java-ns type-map]
   (let [{:keys [structs enums services]} (process-namespace java-ns)
         new-ns (str *ns*)
         type-locator (fn [type-key type-name]
                        (condp = type-key
                          :class java-ns
                          :struct (if-let [match
                                           (->> type-map
                                                (filter (fn [[k v]] (contains? v (keyword type-name))))
                                                first
                                                first)]
                                    (name match)
                                    new-ns)
                          new-ns))]
     `(do
        ~@(clj-requires type-map)
        ~@(clj-structs structs type-locator)
        ~@(clj-enums enums)
        ~@(clj-services services type-locator))))

;; Helper functions

(defn- varspec [thing]
  (variant/variant-spec spec/+no-precondition+ [{:schema thing}]))

(defn- xplain [thing]
  (if (satisfies? s/Schema thing) (s/explain thing) thing))

(defn- builtin-type? [v]
  (boolean ((set (vals jprims)) (class v))))

(defn- primitive-array? [v]
  (get jarrays (class v)))

(def ^:private pass-through? (some-fn nil? map? vector? set? builtin-type? primitive-array?
                                      (partial instance? org.apache.thrift.TEnum)))

(defn- cap-first [s]
  (st/join (conj (rest s) (st/capitalize (first s)))))

(defn- setter-method [kw]
  (str "set" (cap-first (->camelCaseString kw))))

(defn- io [& vals]
  (into-array java.lang.Object vals))

(defn maybe-error [original coerced schema]
  (when-let [error (error-val coerced)]
    (schema.macros/error! (format* "Value cannot be coerced to match schema: %s" (pr-str error))
                          {:schema schema :value original :error error}))
  coerced)

(defn- struct-schema? [schema]
  (and (class? schema) (extends? coerce/ApiStruct schema)))

(defn- typedef-schema? [schema]
  (instance? TType schema))

(defn- typedef? [f]
  (cond
    (and (instance? TType f) (instance? StructFieldType (:type f))) nil
    (instance? TType f) f
    (and (instance? ContainerFieldType f) (= :map (:type f))) (typedef? (second (:value f)))
    (instance? ContainerFieldType f) (typedef? (first (:value f)))
    :default nil))

(defn- maybe [f]
  (let [fs (:type f)]
    (if-not (:required? f) `(s/maybe ~fs) fs)))
