;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.connect
  "Utilities for connecting Thrift servers and clients."
  (:require [camel-snake-kebab.core :refer [->kebab-case-symbol]]
            [thrifty.service :as svc])
  (:import java.lang.Class
           [java.net InetAddress InetSocketAddress]
           [org.apache.thrift TProcessorFactory TServiceClient]
           [org.apache.thrift.protocol TBinaryProtocol TCompactProtocol TJSONProtocol
            TBinaryProtocol$Factory TCompactProtocol$Factory TJSONProtocol$Factory]
           [org.apache.thrift.server TNonblockingServer TServer TThreadPoolServer TThreadPoolServer$Args
            TServer$Args TNonblockingServer$Args TSimpleServer]
           [org.apache.thrift.transport TSocket TSSLTransportFactory TSSLTransportFactory$TSSLTransportParameters
            TServerSocket]
           org.reflections.ReflectionUtils))

(defmacro args-builder
  "Macro that creates a function to build a server's AbstractServerArgs instance.

  The function will take kebab-style keyword arguments corresponding to each available argument.
  Common servers already have their argument functions in `server-args'.

  ex: `((args-builder TThreadPoolServer$Args) transport :min-worker-threads 5)'"
  [^Class args-cls]
  (let [methods (ReflectionUtils/getAllMethods (resolve args-cls) nil)
        method-names (map #(.getName %) methods)
        method-keys (map ->kebab-case-symbol method-names)]
    `(fn [~'transport & {:keys [~@method-keys]}]
       (let [~'args (new ~args-cls ~'transport)]
         ~@(for [name method-names]
             `(when-not (nil? ~(->kebab-case-symbol name))
                (~(symbol (str "." name)) ~'args ~(->kebab-case-symbol name))))
         ~'args))))

(def server-args
  "Args instance functions for common Thrift server types."
  {:threadpool (args-builder TThreadPoolServer$Args)
   :simple (args-builder TServer$Args)
   :async (args-builder TNonblockingServer$Args)})

(def ^:private server-factories {:threadpool #(TThreadPoolServer. %)
                                 :simple #(TSimpleServer. %)
                                 :async #(TNonblockingServer. %)})

(def ^:private protocol-factories {:binary (TBinaryProtocol$Factory.)
                                   :json (TJSONProtocol$Factory.)
                                   :compact (TCompactProtocol$Factory.)})

(defn- ssl-transport-factory [server? opts]
  (let [ssl-opts (:ssl-params opts)
        hostname (.getHostName (:host opts))]
    (if (and ssl-opts (:client-auth opts)) (.requireClientAuth ssl-opts true))
    (if server?
      (if ssl-opts
        (TSSLTransportFactory/getServerSocket (int (:port opts)) (int (:client-timeout opts)) (:host opts) ssl-opts)
        (TSSLTransportFactory/getServerSocket (int (:port opts)) (int (:client-timeout opts))
                                              (:client-auth opts) (:host opts)))
      (if ssl-opts
        (TSSLTransportFactory/getClientSocket hostname (int (:port opts)) (int (:client-timeout opts)) ssl-opts)
        (TSSLTransportFactory/getClientSocket hostname (int (:port opts)) (int (:client-timeout opts)))))))

(defn- processor-factory [opts]
  (TProcessorFactory. (svc/processor (:handler opts))))

(def ^:private defaults {:protocol :binary :client-timeout 0 :client-auth false})

(defn mk-server
  "Create and return a TServer.

  Usage:

  (mk-server handler server-type port host? options?

  where:
    - handler is an implementation of a Service protocol (see: `thrifty.generator.api')
    - server-type is ( :threadpool | :simple | :async )
    - port is an integer port to bind
    - host is an optional host string or `InetAddress', defaulting to localhost
    - options is a map of optional configuration for this function and/or server `Args' instance.
      {:ssl true | false
       :ssl-params `TSSLTransportFactory$TSSLTransportParameters' instance
       :client-timeout Connection timeout in MS
       :client-auth true | false
       :processor-factory fn taking one arg `factory-opts' returning `TProcessorFactory'
       :transport-factory fn taking one arg `factory-opts' returning `TTransport'
       ...}
      All elided arguments can be used by `arg-builder' specific to `server-type'

    factory-opts includes all `options' plus:
      - :inet `InetSocketaddress'
      - :host `InetAddress'
      - :port see above
      - :server-type see above
      - :handler see above"
  ([server-type handler port] (mk-server handler server-type port (InetAddress/getLoopbackAddress)))
  ([server-type handler port host] (mk-server handler server-type port host {}))
  ([server-type handler port host options]
   {:pre [(server-type server-args) (number? port)]}
   (let [hostnet (if-not (instance? InetAddress host) (InetAddress/getByName host) host)
         inet (InetSocketAddress. hostnet port)
         factory-opts (merge defaults {:inet inet :host hostnet :port port :server-type server-type
                                       :handler handler} options)
         protocol-factory ((:protocol factory-opts) protocol-factories)
         transport-factory (cond (:transport-factory options) (:transport-factory options)
                                 (:ssl options) (partial #'ssl-transport-factory true)
                                 :default (fn [_] (TServerSocket. inet)))
         processor-factory (get options :processor-factory #'processor-factory)
         transport (transport-factory factory-opts)
         arg-opts (dissoc factory-opts :protocol-factory :processor-factory :transport-factory)
         args (apply (server-type server-args) (concat [transport] (reduce concat arg-opts)))]
     (doto args
       (.protocolFactory protocol-factory)
       (.processorFactory (processor-factory factory-opts)))
     ((server-type server-factories) args))))

(defn ^TServiceClient connect!
  "Connect a (Closeable) client to a Thrift service.

  Usage and options mirror those of `mk-server', though the Transport factory is expected to return
  `TSocket' for obvious reasons."
  ([handler port] (connect! handler port (InetAddress/getLoopbackAddress)))
  ([handler port host] (connect! handler port host {}))
  ([handler port host options]
   (let [hostnet (if-not (instance? InetAddress host) (InetAddress/getByName host) host)
         factory-opts (merge defaults {:host hostnet :port port :handler handler} options)
         transport-factory (cond (:transport-factory options) (:transport-factory options)
                                 (:ssl options) (partial #'ssl-transport-factory false)
                                 :default (fn [_] (TSocket. host port)))
         transport (transport-factory factory-opts)
         protocol-factory ((:protocol factory-opts) protocol-factories)]
     (svc/client handler (.getProtocol protocol-factory transport)))))

(defn ^TServer serve-and-block!
  "Start server in current thread, blocking indefinitely."
  [^TServer server]
  (.serve server)
  server)

(defn ^TServer serve!
  "Start server in a new thread, returning TServer."
  [^TServer server]
  (future (serve-and-block! server))
  server)
