;; Copyright 2016 Neumitra, Inc.

;; Licensed under the Apache License, Version 2.0 (the "License");
;; you may not use this file except in compliance with the License.
;; You may obtain a copy of the License at

;; http://www.apache.org/licenses/LICENSE-2.0

;; Unless required by applicable law or agreed to in writing, software
;; distributed under the License is distributed on an "AS IS" BASIS,
;; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
;; See the License for the specific language governing permissions and
;; limitations under the License.

(ns thrifty.coerce
  "Contains multimethods and utilities for converting API objects between Clojure and Java."
  (:require [camel-snake-kebab.core :refer [->kebab-case-keyword]]
            [clojure.string :as st]
            [thrifty.parser.schemas]
            [thrifty.reflector :refer [pred]])
  (:import org.apache.thrift.TBase
           org.reflections.ReflectionUtils
           [thrifty.parser.schemas TStruct TType]))

(defprotocol ApiStruct
  "Internal protocol used to work with Thrift structs."
  (struct-schema [this])
  (java-class [this]))

(defn noop
  "Used in place of to-schema/to-domain for a no-op."
  [s o] o)

(defn mapper
  "Given a record y/X, find a mapper function y/map->X"
  ([rec] (mapper rec true))
  ([rec resolve?] (mapper rec resolve? "map->"))
  ([rec resolve? prefix]
   (try
     (let [parts (st/split (.getName rec) #"\.")
           jns (st/join "." (butlast parts))
           ns (st/replace jns "_" "-")
           m (symbol (str ns "/" prefix (last parts)))]
       (if resolve? (resolve m) m))
     (catch java.lang.IllegalArgumentException e nil))))

(defn dispatch
  "Dispatching for to-domain/to-schema.

  - Thrift structs and typedefs are dispatched on [:namespace :name]
  - Everything else is dispatched by class."
  [schema object]
  (cond
    (keyword? schema) schema
    (or (instance? TStruct schema) (instance? TType schema)) (keyword (:ns schema) (:name schema))
    (and (class? schema) (extends? ApiStruct schema)) (dispatch ((mapper schema) {}) object)
    (and (not (class? schema)) (satisfies? ApiStruct schema)) (dispatch (struct-schema schema) object)
    :else (class object)))

(defmulti to-domain
  "Convert an object to is 'domain' representation: the form used in end-user code."
  #'dispatch)
(defmulti to-schema
  "Convert an object to its 'schema' representation: the form that is valid for its `schema' definition."
  #'dispatch)

(defmethod to-domain :default [s o] o)
(defmethod to-schema :default [s o] o)

(defn jc-dispatch [schema object]
  (if (instance? TBase object) TBase (class object)))

(defmulti java->clj #'jc-dispatch)

(defn- fields [cls]
  (let [pred (pred :public)]
    (ReflectionUtils/getAllFields cls pred)))

(def ^:private fields-memo (memoize fields))

(defmethod java->clj java.nio.Buffer [s o] (.array o))
(defmethod java->clj java.util.List [s o] (vec (map (partial java->clj s) o)))
(defmethod java->clj java.util.Set [s o] (set (map (partial java->clj s) o)))
(defmethod java->clj java.util.Map [s o] (into {} o))
(defmethod java->clj java.lang.Boolean [s o] (boolean o))
(defmethod java->clj TBase [s o]
  (let [fields (fields-memo (class o))
        ->pair (fn [i] [(->kebab-case-keyword (.getName i))
                        (java->clj s (.get i o))])]
    (apply dissoc (into {} (map ->pair fields)) #{:meta-data-map})))
(defmethod java->clj :default [s o] o)
