(ns nedap.lacinia.pedestal.multipart.impl.upload
  (:require
   [cheshire.core :as cheshire]
   [clojure.string :as str]
   [nedap.lacinia.pedestal.multipart.impl.content-type :as content-type]))

(defn ^:private mapping-by-part
  "Transform mapping from {k [& vs]} to {v1 k v2 k ...}"
  [mapping]
  (reduce-kv (fn [memo part paths]
               (merge memo (zipmap paths (repeat part))))
             {}
             mapping))

(defn ^:private assoc-path
  [m path v]
  (letfn [(parse-segment [m segment]
            (cond
              (map? m) (keyword segment)
              (vector? m) (Long/parseLong segment)
              :else (throw (ex-info "Unexpected collection in variables" {:m m}))))

          (assoc-segments [m [segment & segments] v]
            (let [k (parse-segment m segment)]
              (if-not segments
                (assoc m k v)
                (assoc m k (assoc-segments (or (get m k)
                                               (throw (ex-info "Malformed path" {:m m :k k})))
                                           segments v)))))]
    (assoc-segments m (str/split path #"\.") v)))

(defn ^:private file-map->apollo
  "Transform a file map to map Apollo's shape"
  [{:keys [tempfile content-type filename]}]
  (let [{params :params mimetype :type} (content-type/parse content-type)]
    (assoc params
           :stream   tempfile
           :mimetype mimetype
           :filename filename
           :encoding "binary")))

(defn handle
  "Merge file uploads referenced in the upload mapping into the variables of the query"
  [request query-map]
  (let [mapping (some-> (get-in request [:params "map"])
                        (cheshire/parse-string)
                        (mapping-by-part))]
    (reduce-kv
     (fn [memo path part]
       (assoc-path memo path (file-map->apollo (or (get-in request [:params part])
                                                   (throw (ex-info "Missing referenced upload part" {:part part}))))))
     query-map
     mapping)))
