(ns nedap.lacinia.pedestal.multipart
  "Extensions for lacinia pedestal to implement the graphql-multipart-request-spec."
  (:require
   [cheshire.core :as cheshire]
   [clojure.set :as set]
   [com.walmartlabs.lacinia.pedestal :as lacinia.pedestal]
   [io.pedestal.http.ring-middlewares :as ring-middlewares]
   [io.pedestal.interceptor :refer [interceptor]]
   [nedap.lacinia.pedestal.multipart.impl.interceptor :as interceptor]
   [nedap.lacinia.pedestal.multipart.impl.upload :as upload]
   [ring.util.request :as request]))

(defmethod lacinia.pedestal/extract-query :multipart/form-data [request]
  (let [operations (get-in request [:params "operations"])
        query-map  (->> (cheshire/parse-string operations true)
                        (upload/handle request))
        renaming   {:query         :graphql-query
                    :variables     :graphql-vars
                    :operationName :graphql-operation-name}]
    (-> query-map
        (set/rename-keys renaming)
        (assoc ::lacinia.pedestal/known-content-type true))))

(defn body-data-interceptor
  "Prepares the body of `request`s for query extraction.

  Serves as a replacement for the non-multipart-aware `#'com.walmartlabs.lacinia.pedestal/body-data-interceptor`"
  ([]
   (body-data-interceptor {}))
  ([multipart-opts]
   (interceptor
    {:name  ::body-interceptor ;; same name as in lacinia.pedestal, but different namespace
     :enter (fn [{:keys [request] :as context}]
              (let [next-interceptor (if (#{"multipart/form-data"} (request/content-type request))
                                       (ring-middlewares/multipart-params multipart-opts)
                                       lacinia.pedestal/body-data-interceptor)]
                ;; While the consumers are only aware of a single interceptor (this one), in
                ;; reality it delegates (immediately) to a second at runtime.
                ;;
                ;; This delegation is functionally equivalent to inlining the implementations of
                ;; the two interceptors being switched on based on content-type.
                (interceptor/prequeue context next-interceptor)))})))
