(ns nedap.lacinia.pedestal.test
  (:require
   [cheshire.core :as cheshire]
   [clojure.string :as string]
   [io.pedestal.test :refer [response-for]]
   [nedap.lacinia.pedestal.multipart.impl.content-type :refer [probe-content-type]])
  (:import
   (java.io ByteArrayInputStream File)
   (java.nio.file Files)))

(defn perform-multipart-query
  "performs a multipart query over the full pedestal stack

  A file can be passed into `variables`"
  [{:keys [url headers query variables service-fn boundary]
    :or {boundary "BOUNDARY"}}]
  (letfn [(flatten-map [path m]
            (if (map? m)
              (mapcat (fn [[k v]] (flatten-map (conj path k) v)) m)
              [[path m]]))

          (filter-vals [pred coll]
            (->> (flatten-map [] coll)
                 (filter (comp pred second))))]
    (let [[[file-path, file] & other-files] (filter-vals #(instance? File %) variables)
          _                 (assert (and (some? file) (nil? other-files)) "`variables should contain 1 file")
          mapping           {"0" [(str "variables." (->> file-path
                                                         (map name)
                                                         (string/join ".")))]}
          filename          (.getName file)
          content-type      (probe-content-type file)
          _                 (assert (some? content-type) "Could not determine content-type of file")

          multipart-heading (->> [(str "--" boundary)
                                  "Content-Disposition: form-data; name=\"operations\""
                                  ""
                                  (cheshire/generate-string {:query     query
                                                             :variables (assoc-in variables file-path nil)})
                                  (str "--" boundary)
                                  "Content-Disposition: form-data; name=\"map\""
                                  ""
                                  (cheshire/generate-string mapping)
                                  (str "--" boundary)
                                  (str "Content-Disposition: form-data; name=\"0\"; filename=" (cheshire/generate-string filename))
                                  (str "Content-Type: " content-type ";charset=utf-8")
                                  "\r\n"]
                                 (string/join "\r\n")
                                 (.getBytes))
          body              (->> (concat multipart-heading
                                         (Files/readAllBytes (.toPath file))
                                         (.getBytes (str "\r\n--" boundary "--")))
                                 (byte-array)
                                 (ByteArrayInputStream.))]
      (-> (response-for service-fn :post url
                        :headers (assoc headers "Content-Type" (str "multipart/form-data; boundary=" boundary))
                        :body body)
          (select-keys [:status :body])
          (update :body cheshire/parse-string keyword)))))
