(ns nedap.lacinia.pedestal.multipart.impl.content-type
  (:require
   [clojure.string :as str]
   [medley.core :refer [assoc-some]])
  (:import
   (java.io File)
   (java.nio.file Files)))

(def ^:private ws "\\s*")

(def ^:private t-special
  (str "("   ")"   "<"   ">"   "@"
       ","   ";"   ":"   "\\"  "\""
       "/"   "\\[" "\\]" "\\?" "\\."
       "="))

(def ^:private any-ws "\\s")

(def ^:private char-or-escape
  (str \( "[^." "\\\\\\\"]" \| "\\\\." \)))

(def ^:private quoted-str
  (str \" char-or-escape "*" \"))

(def ^:private token
  (str "[^" any-ws t-special "]+"))

(def ^:private value
  (str \( token \| quoted-str \)))

(def ^:private param
  (str \; ws "(?<param>" token ")" ws \= ws "(?<value>" value ")" ws))

(def ^:private re-param
  (re-pattern param))

(def ^:private re-content-type
  (re-pattern
   (str ws "(?<type>(" token \/ token "))" ws "(?<params>(" param ")*)")))

(defn ^:private raw-value
  [value]
  (-> value
      (str/replace #"^\"(.*)\"$" "$1")
      (str/replace #"\\(.)" "$1")))

(defn ^:private parse-params
  [params]
  (let [m (re-matcher re-param params)]
    (letfn [(parse-param [memo]
              (if-not (.find m)
                memo
                (recur (assoc memo (keyword (.group m "param"))  (raw-value (.group m "value"))))))]
      (parse-param nil))))

(defn parse
  [content-type]
  (let [matcher (re-matcher re-content-type content-type)]
    (when (.matches matcher)
      (let [type (.group matcher "type")
            params (.group matcher "params")]
        (assoc-some {:type type}
                    :params (parse-params params))))))

(defmacro ^:private when-javax-activation [& body]
  (when (try (Class/forName "javax.activation.MimetypesFileTypeMap")
             (catch Throwable _ nil))
    `(do ~@body)))

;; Files/probeContentType returns nil on macOS with JDK 8 and lower.
;; #'javax.activation.MimetypesFileTypeMap is used as fallback, but that might not be available
(defn probe-content-type  [^File file]
  (or (Files/probeContentType (.toPath file))
      (when-javax-activation
       (.getContentType (javax.activation.MimetypesFileTypeMap.) file))))
